/***********************************************************************

  A JavaScript tokenizer / parser / beautifier / compressor.
  https://github.com/mishoo/UglifyJS2

  -------------------------------- (C) ---------------------------------

                           Author: Mihai Bazon
                         <mihai.bazon@gmail.com>
                       http://mihai.bazon.net/blog

  Distributed under the BSD license:

    Copyright 2012 (c) Mihai Bazon <mihai.bazon@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

        * Redistributions of source code must retain the above
          copyright notice, this list of conditions and the following
          disclaimer.

        * Redistributions in binary form must reproduce the above
          copyright notice, this list of conditions and the following
          disclaimer in the documentation and/or other materials
          provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER “AS IS” AND ANY
    EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
    PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE
    LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
    OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
    PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
    PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
    THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
    THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.

 ***********************************************************************/

"use strict";

import {
    defaults,
    HOP,
    keep_name,
    make_node,
    makePredicate,
    map_add,
    MAP,
    member,
    noop,
    remove,
    return_false,
    return_null,
    return_this,
    return_true,
    string_template,
    regexp_source_fix,
    has_annotation
} from "../utils/index.js";
import { first_in_statement, } from "../utils/first_in_statement.js";
import {
    AST_Accessor,
    AST_Array,
    AST_Arrow,
    AST_Assign,
    AST_Await,
    AST_BigInt,
    AST_Binary,
    AST_Block,
    AST_BlockStatement,
    AST_Boolean,
    AST_Break,
    AST_Call,
    AST_Case,
    AST_Catch,
    AST_Chain,
    AST_Class,
    AST_ClassExpression,
    AST_ClassProperty,
    AST_ConciseMethod,
    AST_Conditional,
    AST_Const,
    AST_Constant,
    AST_Continue,
    AST_Debugger,
    AST_Default,
    AST_DefaultAssign,
    AST_DefClass,
    AST_Definitions,
    AST_Defun,
    AST_Destructuring,
    AST_Directive,
    AST_Do,
    AST_Dot,
    AST_DWLoop,
    AST_EmptyStatement,
    AST_Exit,
    AST_Expansion,
    AST_Export,
    AST_False,
    AST_Finally,
    AST_For,
    AST_ForIn,
    AST_Function,
    AST_Hole,
    AST_If,
    AST_Import,
    AST_Infinity,
    AST_IterationStatement,
    AST_Jump,
    AST_LabeledStatement,
    AST_Lambda,
    AST_Let,
    AST_LoopControl,
    AST_NaN,
    AST_New,
    AST_Node,
    AST_Null,
    AST_Number,
    AST_Object,
    AST_ObjectGetter,
    AST_ObjectKeyVal,
    AST_ObjectProperty,
    AST_ObjectSetter,
    AST_PrefixedTemplateString,
    AST_PropAccess,
    AST_RegExp,
    AST_Return,
    AST_Scope,
    AST_Sequence,
    AST_SimpleStatement,
    AST_Statement,
    AST_String,
    AST_Sub,
    AST_Switch,
    AST_SwitchBranch,
    AST_Symbol,
    AST_SymbolBlockDeclaration,
    AST_SymbolCatch,
    AST_SymbolClassProperty,
    AST_SymbolConst,
    AST_SymbolDeclaration,
    AST_SymbolDefun,
    AST_SymbolExport,
    AST_SymbolFunarg,
    AST_SymbolLambda,
    AST_SymbolLet,
    AST_SymbolMethod,
    AST_SymbolRef,
    AST_SymbolVar,
    AST_TemplateSegment,
    AST_TemplateString,
    AST_This,
    AST_Toplevel,
    AST_True,
    AST_Try,
    AST_Unary,
    AST_UnaryPostfix,
    AST_UnaryPrefix,
    AST_Undefined,
    AST_Var,
    AST_VarDef,
    AST_While,
    AST_With,
    AST_Yield,

    TreeTransformer,
    TreeWalker,
    walk,
    walk_abort,
    walk_body,
    walk_parent,

    _INLINE,
    _NOINLINE,
    _PURE
} from "../ast.js";
import { equivalent_to } from "../equivalent-to.js";
import {
    is_basic_identifier_string,
    JS_Parse_Error,
    parse,
    PRECEDENCE,
} from "../parse.js";
import { OutputStream } from "../output.js";
import {
    base54,
    SymbolDef,
} from "../scope.js";
import "../size.js";

const UNUSED    = 0b00000001;
const TRUTHY    = 0b00000010;
const FALSY     = 0b00000100;
const UNDEFINED = 0b00001000;
const INLINED   = 0b00010000;
// Nodes to which values are ever written. Used when keep_assign is part of the unused option string.
const WRITE_ONLY= 0b00100000;

// information specific to a single compression pass
const SQUEEZED  = 0b0000000100000000;
const OPTIMIZED = 0b0000001000000000;
const TOP       = 0b0000010000000000;

const CLEAR_BETWEEN_PASSES = SQUEEZED | OPTIMIZED | TOP;

const has_flag = (node, flag) => node.flags & flag;
const set_flag = (node, flag) => { node.flags |= flag; };
const clear_flag = (node, flag) => { node.flags &= ~flag; };

class Compressor extends TreeWalker {
    constructor(options, { false_by_default = false, mangle_options = false }) {
        super();
        if (options.defaults !== undefined && !options.defaults) false_by_default = true;
        this.options = defaults(options, {
            arguments     : false,
            arrows        : !false_by_default,
            booleans      : !false_by_default,
            booleans_as_integers : false,
            collapse_vars : !false_by_default,
            comparisons   : !false_by_default,
            computed_props: !false_by_default,
            conditionals  : !false_by_default,
            dead_code     : !false_by_default,
            defaults      : true,
            directives    : !false_by_default,
            drop_console  : false,
            drop_debugger : !false_by_default,
            ecma          : 5,
            evaluate      : !false_by_default,
            expression    : false,
            global_defs   : false,
            hoist_funs    : false,
            hoist_props   : !false_by_default,
            hoist_vars    : false,
            ie8           : false,
            if_return     : !false_by_default,
            inline        : !false_by_default,
            join_vars     : !false_by_default,
            keep_classnames: false,
            keep_fargs    : true,
            keep_fnames   : false,
            keep_infinity : false,
            loops         : !false_by_default,
            module        : false,
            negate_iife   : !false_by_default,
            passes        : 1,
            properties    : !false_by_default,
            pure_getters  : !false_by_default && "strict",
            pure_funcs    : null,
            reduce_funcs  : !false_by_default,
            reduce_vars   : !false_by_default,
            sequences     : !false_by_default,
            side_effects  : !false_by_default,
            switches      : !false_by_default,
            top_retain    : null,
            toplevel      : !!(options && options["top_retain"]),
            typeofs       : !false_by_default,
            unsafe        : false,
            unsafe_arrows : false,
            unsafe_comps  : false,
            unsafe_Function: false,
            unsafe_math   : false,
            unsafe_symbols: false,
            unsafe_methods: false,
            unsafe_proto  : false,
            unsafe_regexp : false,
            unsafe_undefined: false,
            unused        : !false_by_default,
            warnings      : false  // legacy
        }, true);
        var global_defs = this.options["global_defs"];
        if (typeof global_defs == "object") for (var key in global_defs) {
            if (key[0] === "@" && HOP(global_defs, key)) {
                global_defs[key.slice(1)] = parse(global_defs[key], {
                    expression: true
                });
            }
        }
        if (this.options["inline"] === true) this.options["inline"] = 3;
        var pure_funcs = this.options["pure_funcs"];
        if (typeof pure_funcs == "function") {
            this.pure_funcs = pure_funcs;
        } else {
            this.pure_funcs = pure_funcs ? function(node) {
                return !pure_funcs.includes(node.expression.print_to_string());
            } : return_true;
        }
        var top_retain = this.options["top_retain"];
        if (top_retain instanceof RegExp) {
            this.top_retain = function(def) {
                return top_retain.test(def.name);
            };
        } else if (typeof top_retain == "function") {
            this.top_retain = top_retain;
        } else if (top_retain) {
            if (typeof top_retain == "string") {
                top_retain = top_retain.split(/,/);
            }
            this.top_retain = function(def) {
                return top_retain.includes(def.name);
            };
        }
        if (this.options["module"]) {
            this.directives["use strict"] = true;
            this.options["toplevel"] = true;
        }
        var toplevel = this.options["toplevel"];
        this.toplevel = typeof toplevel == "string" ? {
            funcs: /funcs/.test(toplevel),
            vars: /vars/.test(toplevel)
        } : {
            funcs: toplevel,
            vars: toplevel
        };
        var sequences = this.options["sequences"];
        this.sequences_limit = sequences == 1 ? 800 : sequences | 0;
        this.evaluated_regexps = new Map();
        this._toplevel = undefined;
        this.mangle_options = mangle_options;
    }

    option(key) {
        return this.options[key];
    }

    exposed(def) {
        if (def.export) return true;
        if (def.global) for (var i = 0, len = def.orig.length; i < len; i++)
            if (!this.toplevel[def.orig[i] instanceof AST_SymbolDefun ? "funcs" : "vars"])
                return true;
        return false;
    }

    in_boolean_context() {
        if (!this.option("booleans")) return false;
        var self = this.self();
        for (var i = 0, p; p = this.parent(i); i++) {
            if (p instanceof AST_SimpleStatement
                || p instanceof AST_Conditional && p.condition === self
                || p instanceof AST_DWLoop && p.condition === self
                || p instanceof AST_For && p.condition === self
                || p instanceof AST_If && p.condition === self
                || p instanceof AST_UnaryPrefix && p.operator == "!" && p.expression === self) {
                return true;
            }
            if (
                p instanceof AST_Binary
                    && (
                        p.operator == "&&"
                        || p.operator == "||"
                        || p.operator == "??"
                    )
                || p instanceof AST_Conditional
                || p.tail_node() === self
            ) {
                self = p;
            } else {
                return false;
            }
        }
    }

    get_toplevel() {
        return this._toplevel;
    }

    compress(toplevel) {
        toplevel = toplevel.resolve_defines(this);
        this._toplevel = toplevel;
        if (this.option("expression")) {
            this._toplevel.process_expression(true);
        }
        var passes = +this.options.passes || 1;
        var min_count = 1 / 0;
        var stopping = false;
        var mangle = { ie8: this.option("ie8") };
        for (var pass = 0; pass < passes; pass++) {
            this._toplevel.figure_out_scope(mangle);
            if (pass === 0 && this.option("drop_console")) {
                // must be run before reduce_vars and compress pass
                this._toplevel = this._toplevel.drop_console();
            }
            if (pass > 0 || this.option("reduce_vars")) {
                this._toplevel.reset_opt_flags(this);
            }
            this._toplevel = this._toplevel.transform(this);
            if (passes > 1) {
                let count = 0;
                walk(this._toplevel, () => { count++; });
                if (count < min_count) {
                    min_count = count;
                    stopping = false;
                } else if (stopping) {
                    break;
                } else {
                    stopping = true;
                }
            }
        }
        if (this.option("expression")) {
            this._toplevel.process_expression(false);
        }
        toplevel = this._toplevel;
        this._toplevel = undefined;
        return toplevel;
    }

    before(node, descend) {
        if (has_flag(node, SQUEEZED)) return node;
        var was_scope = false;
        if (node instanceof AST_Scope) {
            node = node.hoist_properties(this);
            node = node.hoist_declarations(this);
            was_scope = true;
        }
        // Before https://github.com/mishoo/UglifyJS2/pull/1602 AST_Node.optimize()
        // would call AST_Node.transform() if a different instance of AST_Node is
        // produced after def_optimize().
        // This corrupts TreeWalker.stack, which cause AST look-ups to malfunction.
        // Migrate and defer all children's AST_Node.transform() to below, which
        // will now happen after this parent AST_Node has been properly substituted
        // thus gives a consistent AST snapshot.
        descend(node, this);
        // Existing code relies on how AST_Node.optimize() worked, and omitting the
        // following replacement call would result in degraded efficiency of both
        // output and performance.
        descend(node, this);
        var opt = node.optimize(this);
        if (was_scope && opt instanceof AST_Scope) {
            opt.drop_unused(this);
            descend(opt, this);
        }
        if (opt === node) set_flag(opt, SQUEEZED);
        return opt;
    }
}

function def_optimize(node, optimizer) {
    node.DEFMETHOD("optimize", function(compressor) {
        var self = this;
        if (has_flag(self, OPTIMIZED)) return self;
        if (compressor.has_directive("use asm")) return self;
        var opt = optimizer(self, compressor);
        set_flag(opt, OPTIMIZED);
        return opt;
    });
}

def_optimize(AST_Node, function(self) {
    return self;
});

AST_Toplevel.DEFMETHOD("drop_console", function() {
    return this.transform(new TreeTransformer(function(self) {
        if (self.TYPE == "Call") {
            var exp = self.expression;
            if (exp instanceof AST_PropAccess) {
                var name = exp.expression;
                while (name.expression) {
                    name = name.expression;
                }
                if (is_undeclared_ref(name) && name.name == "console") {
                    return make_node(AST_Undefined, self);
                }
            }
        }
    }));
});

AST_Node.DEFMETHOD("equivalent_to", function(node) {
    return equivalent_to(this, node);
});

AST_Scope.DEFMETHOD("process_expression", function(insert, compressor) {
    var self = this;
    var tt = new TreeTransformer(function(node) {
        if (insert && node instanceof AST_SimpleStatement) {
            return make_node(AST_Return, node, {
                value: node.body
            });
        }
        if (!insert && node instanceof AST_Return) {
            if (compressor) {
                var value = node.value && node.value.drop_side_effect_free(compressor, true);
                return value ? make_node(AST_SimpleStatement, node, {
                    body: value
                }) : make_node(AST_EmptyStatement, node);
            }
            return make_node(AST_SimpleStatement, node, {
                body: node.value || make_node(AST_UnaryPrefix, node, {
                    operator: "void",
                    expression: make_node(AST_Number, node, {
                        value: 0
                    })
                })
            });
        }
        if (node instanceof AST_Class || node instanceof AST_Lambda && node !== self) {
            return node;
        }
        if (node instanceof AST_Block) {
            var index = node.body.length - 1;
            if (index >= 0) {
                node.body[index] = node.body[index].transform(tt);
            }
        } else if (node instanceof AST_If) {
            node.body = node.body.transform(tt);
            if (node.alternative) {
                node.alternative = node.alternative.transform(tt);
            }
        } else if (node instanceof AST_With) {
            node.body = node.body.transform(tt);
        }
        return node;
    });
    self.transform(tt);
});

function read_property(obj, key) {
    key = get_value(key);
    if (key instanceof AST_Node) return;

    var value;
    if (obj instanceof AST_Array) {
        var elements = obj.elements;
        if (key == "length") return make_node_from_constant(elements.length, obj);
        if (typeof key == "number" && key in elements) value = elements[key];
    } else if (obj instanceof AST_Object) {
        key = "" + key;
        var props = obj.properties;
        for (var i = props.length; --i >= 0;) {
            var prop = props[i];
            if (!(prop instanceof AST_ObjectKeyVal)) return;
            if (!value && props[i].key === key) value = props[i].value;
        }
    }

    return value instanceof AST_SymbolRef && value.fixed_value() || value;
}

function is_modified(compressor, tw, node, value, level, immutable) {
    var parent = tw.parent(level);
    var lhs = is_lhs(node, parent);
    if (lhs) return lhs;
    if (!immutable
        && parent instanceof AST_Call
        && parent.expression === node
        && !(value instanceof AST_Arrow)
        && !(value instanceof AST_Class)
        && !parent.is_callee_pure(compressor)
        && (!(value instanceof AST_Function)
            || !(parent instanceof AST_New) && value.contains_this())) {
        return true;
    }
    if (parent instanceof AST_Array) {
        return is_modified(compressor, tw, parent, parent, level + 1);
    }
    if (parent instanceof AST_ObjectKeyVal && node === parent.value) {
        var obj = tw.parent(level + 1);
        return is_modified(compressor, tw, obj, obj, level + 2);
    }
    if (parent instanceof AST_PropAccess && parent.expression === node) {
        var prop = read_property(value, parent.property);
        return !immutable && is_modified(compressor, tw, parent, prop, level + 1);
    }
}

(function(def_reduce_vars) {
    def_reduce_vars(AST_Node, noop);

    function reset_def(compressor, def) {
        def.assignments = 0;
        def.chained = false;
        def.direct_access = false;
        def.escaped = 0;
        def.recursive_refs = 0;
        def.references = [];
        def.single_use = undefined;
        if (def.scope.pinned()) {
            def.fixed = false;
        } else if (def.orig[0] instanceof AST_SymbolConst || !compressor.exposed(def)) {
            def.fixed = def.init;
        } else {
            def.fixed = false;
        }
    }

    function reset_variables(tw, compressor, node) {
        node.variables.forEach(function(def) {
            reset_def(compressor, def);
            if (def.fixed === null) {
                tw.defs_to_safe_ids.set(def.id, tw.safe_ids);
                mark(tw, def, true);
            } else if (def.fixed) {
                tw.loop_ids.set(def.id, tw.in_loop);
                mark(tw, def, true);
            }
        });
    }

    function reset_block_variables(compressor, node) {
        if (node.block_scope) node.block_scope.variables.forEach((def) => {
            reset_def(compressor, def);
        });
    }

    function push(tw) {
        tw.safe_ids = Object.create(tw.safe_ids);
    }

    function pop(tw) {
        tw.safe_ids = Object.getPrototypeOf(tw.safe_ids);
    }

    function mark(tw, def, safe) {
        tw.safe_ids[def.id] = safe;
    }

    function safe_to_read(tw, def) {
        if (def.single_use == "m") return false;
        if (tw.safe_ids[def.id]) {
            if (def.fixed == null) {
                var orig = def.orig[0];
                if (orig instanceof AST_SymbolFunarg || orig.name == "arguments") return false;
                def.fixed = make_node(AST_Undefined, orig);
            }
            return true;
        }
        return def.fixed instanceof AST_Defun;
    }

    function safe_to_assign(tw, def, scope, value) {
        if (def.fixed === undefined) return true;
        let def_safe_ids;
        if (def.fixed === null
            && (def_safe_ids = tw.defs_to_safe_ids.get(def.id))
        ) {
            def_safe_ids[def.id] = false;
            tw.defs_to_safe_ids.delete(def.id);
            return true;
        }
        if (!HOP(tw.safe_ids, def.id)) return false;
        if (!safe_to_read(tw, def)) return false;
        if (def.fixed === false) return false;
        if (def.fixed != null && (!value || def.references.length > def.assignments)) return false;
        if (def.fixed instanceof AST_Defun) {
            return value instanceof AST_Node && def.fixed.parent_scope === scope;
        }
        return def.orig.every((sym) => {
            return !(sym instanceof AST_SymbolConst
                || sym instanceof AST_SymbolDefun
                || sym instanceof AST_SymbolLambda);
        });
    }

    function ref_once(tw, compressor, def) {
        return compressor.option("unused")
            && !def.scope.pinned()
            && def.references.length - def.recursive_refs == 1
            && tw.loop_ids.get(def.id) === tw.in_loop;
    }

    function is_immutable(value) {
        if (!value) return false;
        return value.is_constant()
            || value instanceof AST_Lambda
            || value instanceof AST_This;
    }

    // A definition "escapes" when its value can leave the point of use.
    // Example: `a = b || c`
    // In this example, "b" and "c" are escaping, because they're going into "a"
    //
    // def.escaped is != 0 when it escapes.
    //
    // When greater than 1, it means that N chained properties will be read off
    // of that def before an escape occurs. This is useful for evaluating
    // property accesses, where you need to know when to stop.
    function mark_escaped(tw, d, scope, node, value, level = 0, depth = 1) {
        var parent = tw.parent(level);
        if (value) {
            if (value.is_constant()) return;
            if (value instanceof AST_ClassExpression) return;
        }

        if (
            parent instanceof AST_Assign && (parent.operator === "=" || parent.logical) && node === parent.right
            || parent instanceof AST_Call && (node !== parent.expression || parent instanceof AST_New)
            || parent instanceof AST_Exit && node === parent.value && node.scope !== d.scope
            || parent instanceof AST_VarDef && node === parent.value
            || parent instanceof AST_Yield && node === parent.value && node.scope !== d.scope
        ) {
            if (depth > 1 && !(value && value.is_constant_expression(scope))) depth = 1;
            if (!d.escaped || d.escaped > depth) d.escaped = depth;
            return;
        } else if (
            parent instanceof AST_Array
            || parent instanceof AST_Await
            || parent instanceof AST_Binary && lazy_op.has(parent.operator)
            || parent instanceof AST_Conditional && node !== parent.condition
            || parent instanceof AST_Expansion
            || parent instanceof AST_Sequence && node === parent.tail_node()
        ) {
            mark_escaped(tw, d, scope, parent, parent, level + 1, depth);
        } else if (parent instanceof AST_ObjectKeyVal && node === parent.value) {
            var obj = tw.parent(level + 1);

            mark_escaped(tw, d, scope, obj, obj, level + 2, depth);
        } else if (parent instanceof AST_PropAccess && node === parent.expression) {
            value = read_property(value, parent.property);

            mark_escaped(tw, d, scope, parent, value, level + 1, depth + 1);
            if (value) return;
        }

        if (level > 0) return;
        if (parent instanceof AST_Sequence && node !== parent.tail_node()) return;
        if (parent instanceof AST_SimpleStatement) return;

        d.direct_access = true;
    }

    const suppress = node => walk(node, node => {
        if (!(node instanceof AST_Symbol)) return;
        var d = node.definition();
        if (!d) return;
        if (node instanceof AST_SymbolRef) d.references.push(node);
        d.fixed = false;
    });
    def_reduce_vars(AST_Accessor, function(tw, descend, compressor) {
        push(tw);
        reset_variables(tw, compressor, this);
        descend();
        pop(tw);
        return true;
    });
    def_reduce_vars(AST_Assign, function(tw, descend, compressor) {
        var node = this;
        if (node.left instanceof AST_Destructuring) {
            suppress(node.left);
            return;
        }

        const finish_walk = () => {
            if (node.logical) {
                node.left.walk(tw);

                push(tw);
                node.right.walk(tw);
                pop(tw);

                return true;
            }
        };

        var sym = node.left;
        if (!(sym instanceof AST_SymbolRef)) return finish_walk();

        var def = sym.definition();
        var safe = safe_to_assign(tw, def, sym.scope, node.right);
        def.assignments++;
        if (!safe) return finish_walk();

        var fixed = def.fixed;
        if (!fixed && node.operator != "=" && !node.logical) return finish_walk();

        var eq = node.operator == "=";
        var value = eq ? node.right : node;
        if (is_modified(compressor, tw, node, value, 0)) return finish_walk();

        def.references.push(sym);

        if (!node.logical) {
            if (!eq) def.chained = true;

            def.fixed = eq ? function() {
                return node.right;
            } : function() {
                return make_node(AST_Binary, node, {
                    operator: node.operator.slice(0, -1),
                    left: fixed instanceof AST_Node ? fixed : fixed(),
                    right: node.right
                });
            };
        }

        if (node.logical) {
            mark(tw, def, false);
            push(tw);
            node.right.walk(tw);
            pop(tw);
            return true;
        }

        mark(tw, def, false);
        node.right.walk(tw);
        mark(tw, def, true);

        mark_escaped(tw, def, sym.scope, node, value, 0, 1);

        return true;
    });
    def_reduce_vars(AST_Binary, function(tw) {
        if (!lazy_op.has(this.operator)) return;
        this.left.walk(tw);
        push(tw);
        this.right.walk(tw);
        pop(tw);
        return true;
    });
    def_reduce_vars(AST_Block, function(tw, descend, compressor) {
        reset_block_variables(compressor, this);
    });
    def_reduce_vars(AST_Case, function(tw) {
        push(tw);
        this.expression.walk(tw);
        pop(tw);
        push(tw);
        walk_body(this, tw);
        pop(tw);
        return true;
    });
    def_reduce_vars(AST_Class, function(tw, descend) {
        clear_flag(this, INLINED);
        push(tw);
        descend();
        pop(tw);
        return true;
    });
    def_reduce_vars(AST_Conditional, function(tw) {
        this.condition.walk(tw);
        push(tw);
        this.consequent.walk(tw);
        pop(tw);
        push(tw);
        this.alternative.walk(tw);
        pop(tw);
        return true;
    });

    def_reduce_vars(AST_Chain, function(tw, descend) {
        // Chains' conditions apply left-to-right, cumulatively.
        // If we walk normally we don't go in that order because we would pop before pushing again
        // Solution: AST_PropAccess and AST_Call push when they are optional, and never pop.
        // Then we pop everything when they are done being walked.
        const safe_ids = tw.safe_ids;

        descend();

        // Unroll back to start
        tw.safe_ids = safe_ids;
        return true;
    });
    def_reduce_vars(AST_Call, function (tw) {
        this.expression.walk(tw);

        if (this.optional) {
            // Never pop -- it's popped at AST_Chain above
            push(tw);
        }

        for (const arg of this.args) arg.walk(tw);

        return true;
    });
    def_reduce_vars(AST_PropAccess, function (tw) {
        if (!this.optional) return;

        this.expression.walk(tw);

        // Never pop -- it's popped at AST_Chain above
        push(tw);

        if (this.property instanceof AST_Node) this.property.walk(tw);

        return true;
    });
    def_reduce_vars(AST_Default, function(tw, descend) {
        push(tw);
        descend();
        pop(tw);
        return true;
    });
    function mark_lambda(tw, descend, compressor) {
        clear_flag(this, INLINED);
        push(tw);
        reset_variables(tw, compressor, this);
        if (this.uses_arguments) {
            descend();
            pop(tw);
            return;
        }
        var iife;
        if (!this.name
            && (iife = tw.parent()) instanceof AST_Call
            && iife.expression === this
            && !iife.args.some(arg => arg instanceof AST_Expansion)
            && this.argnames.every(arg_name => arg_name instanceof AST_Symbol)
        ) {
            // Virtually turn IIFE parameters into variable definitions:
            //   (function(a,b) {...})(c,d) => (function() {var a=c,b=d; ...})()
            // So existing transformation rules can work on them.
            this.argnames.forEach((arg, i) => {
                if (!arg.definition) return;
                var d = arg.definition();
                // Avoid setting fixed when there's more than one origin for a variable value
                if (d.orig.length > 1) return;
                if (d.fixed === undefined && (!this.uses_arguments || tw.has_directive("use strict"))) {
                    d.fixed = function() {
                        return iife.args[i] || make_node(AST_Undefined, iife);
                    };
                    tw.loop_ids.set(d.id, tw.in_loop);
                    mark(tw, d, true);
                } else {
                    d.fixed = false;
                }
            });
        }
        descend();
        pop(tw);
        return true;
    }

    def_reduce_vars(AST_Lambda, mark_lambda);

    def_reduce_vars(AST_Do, function(tw, descend, compressor) {
        reset_block_variables(compressor, this);
        const saved_loop = tw.in_loop;
        tw.in_loop = this;
        push(tw);
        this.body.walk(tw);
        if (has_break_or_continue(this)) {
            pop(tw);
            push(tw);
        }
        this.condition.walk(tw);
        pop(tw);
        tw.in_loop = saved_loop;
        return true;
    });
    def_reduce_vars(AST_For, function(tw, descend, compressor) {
        reset_block_variables(compressor, this);
        if (this.init) this.init.walk(tw);
        const saved_loop = tw.in_loop;
        tw.in_loop = this;
        push(tw);
        if (this.condition) this.condition.walk(tw);
        this.body.walk(tw);
        if (this.step) {
            if (has_break_or_continue(this)) {
                pop(tw);
                push(tw);
            }
            this.step.walk(tw);
        }
        pop(tw);
        tw.in_loop = saved_loop;
        return true;
    });
    def_reduce_vars(AST_ForIn, function(tw, descend, compressor) {
        reset_block_variables(compressor, this);
        suppress(this.init);
        this.object.walk(tw);
        const saved_loop = tw.in_loop;
        tw.in_loop = this;
        push(tw);
        this.body.walk(tw);
        pop(tw);
        tw.in_loop = saved_loop;
        return true;
    });

    def_reduce_vars(AST_If, function(tw) {
        this.condition.walk(tw);
        push(tw);
        this.body.walk(tw);
        pop(tw);
        if (this.alternative) {
            push(tw);
            this.alternative.walk(tw);
            pop(tw);
        }
        return true;
    });
    def_reduce_vars(AST_LabeledStatement, function(tw) {
        push(tw);
        this.body.walk(tw);
        pop(tw);
        return true;
    });
    def_reduce_vars(AST_SymbolCatch, function() {
        this.definition().fixed = false;
    });

    def_reduce_vars(AST_SymbolRef, function(tw, descend, compressor) {
        var d = this.definition();
        d.references.push(this);
        if (d.references.length == 1
            && !d.fixed
            && d.orig[0] instanceof AST_SymbolDefun) {
            tw.loop_ids.set(d.id, tw.in_loop);
        }
        var fixed_value;
        if (d.fixed === undefined || !safe_to_read(tw, d)) {
            d.fixed = false;
        } else if (d.fixed) {
            fixed_value = this.fixed_value();
            if (
                fixed_value instanceof AST_Lambda
                && recursive_ref(tw, d)
            ) {
                d.recursive_refs++;
            } else if (fixed_value
                && !compressor.exposed(d)
                && ref_once(tw, compressor, d)
            ) {
                d.single_use =
                    fixed_value instanceof AST_Lambda && !fixed_value.pinned()
                    || fixed_value instanceof AST_Class
                    || d.scope === this.scope && fixed_value.is_constant_expression();
            } else {
                d.single_use = false;
            }
            if (is_modified(compressor, tw, this, fixed_value, 0, is_immutable(fixed_value))) {
                if (d.single_use) {
                    d.single_use = "m";
                } else {
                    d.fixed = false;
                }
            }
        }
        mark_escaped(tw, d, this.scope, this, fixed_value, 0, 1);
    });
    def_reduce_vars(AST_Toplevel, function(tw, descend, compressor) {
        this.globals.forEach(function(def) {
            reset_def(compressor, def);
        });
        reset_variables(tw, compressor, this);
    });
    def_reduce_vars(AST_Try, function(tw, descend, compressor) {
        reset_block_variables(compressor, this);
        push(tw);
        walk_body(this, tw);
        pop(tw);
        if (this.bcatch) {
            push(tw);
            this.bcatch.walk(tw);
            pop(tw);
        }
        if (this.bfinally) this.bfinally.walk(tw);
        return true;
    });
    def_reduce_vars(AST_Unary, function(tw) {
        var node = this;
        if (node.operator !== "++" && node.operator !== "--") return;
        var exp = node.expression;
        if (!(exp instanceof AST_SymbolRef)) return;
        var def = exp.definition();
        var safe = safe_to_assign(tw, def, exp.scope, true);
        def.assignments++;
        if (!safe) return;
        var fixed = def.fixed;
        if (!fixed) return;
        def.references.push(exp);
        def.chained = true;
        def.fixed = function() {
            return make_node(AST_Binary, node, {
                operator: node.operator.slice(0, -1),
                left: make_node(AST_UnaryPrefix, node, {
                    operator: "+",
                    expression: fixed instanceof AST_Node ? fixed : fixed()
                }),
                right: make_node(AST_Number, node, {
                    value: 1
                })
            });
        };
        mark(tw, def, true);
        return true;
    });
    def_reduce_vars(AST_VarDef, function(tw, descend) {
        var node = this;
        if (node.name instanceof AST_Destructuring) {
            suppress(node.name);
            return;
        }
        var d = node.name.definition();
        if (node.value) {
            if (safe_to_assign(tw, d, node.name.scope, node.value)) {
                d.fixed = function() {
                    return node.value;
                };
                tw.loop_ids.set(d.id, tw.in_loop);
                mark(tw, d, false);
                descend();
                mark(tw, d, true);
                return true;
            } else {
                d.fixed = false;
            }
        }
    });
    def_reduce_vars(AST_While, function(tw, descend, compressor) {
        reset_block_variables(compressor, this);
        const saved_loop = tw.in_loop;
        tw.in_loop = this;
        push(tw);
        descend();
        pop(tw);
        tw.in_loop = saved_loop;
        return true;
    });
})(function(node, func) {
    node.DEFMETHOD("reduce_vars", func);
});

AST_Toplevel.DEFMETHOD("reset_opt_flags", function(compressor) {
    const self = this;
    const reduce_vars = compressor.option("reduce_vars");

    const preparation = new TreeWalker(function(node, descend) {
        clear_flag(node, CLEAR_BETWEEN_PASSES);
        if (reduce_vars) {
            if (compressor.top_retain
                && node instanceof AST_Defun  // Only functions are retained
                && preparation.parent() === self
            ) {
                set_flag(node, TOP);
            }
            return node.reduce_vars(preparation, descend, compressor);
        }
    });
    // Stack of look-up tables to keep track of whether a `SymbolDef` has been
    // properly assigned before use:
    // - `push()` & `pop()` when visiting conditional branches
    preparation.safe_ids = Object.create(null);
    preparation.in_loop = null;
    preparation.loop_ids = new Map();
    preparation.defs_to_safe_ids = new Map();
    self.walk(preparation);
});

AST_Symbol.DEFMETHOD("fixed_value", function() {
    var fixed = this.thedef.fixed;
    if (!fixed || fixed instanceof AST_Node) return fixed;
    return fixed();
});

AST_SymbolRef.DEFMETHOD("is_immutable", function() {
    var orig = this.definition().orig;
    return orig.length == 1 && orig[0] instanceof AST_SymbolLambda;
});

function is_func_expr(node) {
    return node instanceof AST_Arrow || node instanceof AST_Function;
}

function is_lhs_read_only(lhs) {
    if (lhs instanceof AST_This) return true;
    if (lhs instanceof AST_SymbolRef) return lhs.definition().orig[0] instanceof AST_SymbolLambda;
    if (lhs instanceof AST_PropAccess) {
        lhs = lhs.expression;
        if (lhs instanceof AST_SymbolRef) {
            if (lhs.is_immutable()) return false;
            lhs = lhs.fixed_value();
        }
        if (!lhs) return true;
        if (lhs instanceof AST_RegExp) return false;
        if (lhs instanceof AST_Constant) return true;
        return is_lhs_read_only(lhs);
    }
    return false;
}

function is_ref_of(ref, type) {
    if (!(ref instanceof AST_SymbolRef)) return false;
    var orig = ref.definition().orig;
    for (var i = orig.length; --i >= 0;) {
        if (orig[i] instanceof type) return true;
    }
}

function find_scope(tw) {
    for (let i = 0;;i++) {
        const p = tw.parent(i);
        if (p instanceof AST_Toplevel) return p;
        if (p instanceof AST_Lambda) return p;
        if (p.block_scope) return p.block_scope;
    }
}

function find_variable(compressor, name) {
    var scope, i = 0;
    while (scope = compressor.parent(i++)) {
        if (scope instanceof AST_Scope) break;
        if (scope instanceof AST_Catch && scope.argname) {
            scope = scope.argname.definition().scope;
            break;
        }
    }
    return scope.find_variable(name);
}

function make_sequence(orig, expressions) {
    if (expressions.length == 1) return expressions[0];
    if (expressions.length == 0) throw new Error("trying to create a sequence with length zero!");
    return make_node(AST_Sequence, orig, {
        expressions: expressions.reduce(merge_sequence, [])
    });
}

function make_node_from_constant(val, orig) {
    switch (typeof val) {
      case "string":
        return make_node(AST_String, orig, {
            value: val
        });
      case "number":
        if (isNaN(val)) return make_node(AST_NaN, orig);
        if (isFinite(val)) {
            return 1 / val < 0 ? make_node(AST_UnaryPrefix, orig, {
                operator: "-",
                expression: make_node(AST_Number, orig, { value: -val })
            }) : make_node(AST_Number, orig, { value: val });
        }
        return val < 0 ? make_node(AST_UnaryPrefix, orig, {
            operator: "-",
            expression: make_node(AST_Infinity, orig)
        }) : make_node(AST_Infinity, orig);
      case "boolean":
        return make_node(val ? AST_True : AST_False, orig);
      case "undefined":
        return make_node(AST_Undefined, orig);
      default:
        if (val === null) {
            return make_node(AST_Null, orig, { value: null });
        }
        if (val instanceof RegExp) {
            return make_node(AST_RegExp, orig, {
                value: {
                    source: regexp_source_fix(val.source),
                    flags: val.flags
                }
            });
        }
        throw new Error(string_template("Can't handle constant of type: {type}", {
            type: typeof val
        }));
    }
}

// we shouldn't compress (1,func)(something) to
// func(something) because that changes the meaning of
// the func (becomes lexical instead of global).
function maintain_this_binding(parent, orig, val) {
    if (parent instanceof AST_UnaryPrefix && parent.operator == "delete"
        || parent instanceof AST_Call && parent.expression === orig
            && (val instanceof AST_PropAccess || val instanceof AST_SymbolRef && val.name == "eval")) {
        return make_sequence(orig, [ make_node(AST_Number, orig, { value: 0 }), val ]);
    }
    return val;
}

function merge_sequence(array, node) {
    if (node instanceof AST_Sequence) {
        array.push(...node.expressions);
    } else {
        array.push(node);
    }
    return array;
}

function as_statement_array(thing) {
    if (thing === null) return [];
    if (thing instanceof AST_BlockStatement) return thing.body;
    if (thing instanceof AST_EmptyStatement) return [];
    if (thing instanceof AST_Statement) return [ thing ];
    throw new Error("Can't convert thing to statement array");
}

function is_empty(thing) {
    if (thing === null) return true;
    if (thing instanceof AST_EmptyStatement) return true;
    if (thing instanceof AST_BlockStatement) return thing.body.length == 0;
    return false;
}

function can_be_evicted_from_block(node) {
    return !(
        node instanceof AST_DefClass ||
        node instanceof AST_Defun ||
        node instanceof AST_Let ||
        node instanceof AST_Const ||
        node instanceof AST_Export ||
        node instanceof AST_Import
    );
}

function loop_body(x) {
    if (x instanceof AST_IterationStatement) {
        return x.body instanceof AST_BlockStatement ? x.body : x;
    }
    return x;
}

function is_iife_call(node) {
    // Used to determine whether the node can benefit from negation.
    // Not the case with arrow functions (you need an extra set of parens).
    if (node.TYPE != "Call") return false;
    return node.expression instanceof AST_Function || is_iife_call(node.expression);
}

function is_undeclared_ref(node) {
    return node instanceof AST_SymbolRef && node.definition().undeclared;
}

var global_names = makePredicate("Array Boolean clearInterval clearTimeout console Date decodeURI decodeURIComponent encodeURI encodeURIComponent Error escape eval EvalError Function isFinite isNaN JSON Math Number parseFloat parseInt RangeError ReferenceError RegExp Object setInterval setTimeout String SyntaxError TypeError unescape URIError");
AST_SymbolRef.DEFMETHOD("is_declared", function(compressor) {
    return !this.definition().undeclared
        || compressor.option("unsafe") && global_names.has(this.name);
});

var identifier_atom = makePredicate("Infinity NaN undefined");
function is_identifier_atom(node) {
    return node instanceof AST_Infinity
        || node instanceof AST_NaN
        || node instanceof AST_Undefined;
}

// Tighten a bunch of statements together. Used whenever there is a block.
function tighten_body(statements, compressor) {
    var in_loop, in_try;
    var scope = compressor.find_parent(AST_Scope).get_defun_scope();
    find_loop_scope_try();
    var CHANGED, max_iter = 10;
    do {
        CHANGED = false;
        eliminate_spurious_blocks(statements);
        if (compressor.option("dead_code")) {
            eliminate_dead_code(statements, compressor);
        }
        if (compressor.option("if_return")) {
            handle_if_return(statements, compressor);
        }
        if (compressor.sequences_limit > 0) {
            sequencesize(statements, compressor);
            sequencesize_2(statements, compressor);
        }
        if (compressor.option("join_vars")) {
            join_consecutive_vars(statements);
        }
        if (compressor.option("collapse_vars")) {
            collapse(statements, compressor);
        }
    } while (CHANGED && max_iter-- > 0);

    function find_loop_scope_try() {
        var node = compressor.self(), level = 0;
        do {
            if (node instanceof AST_Catch || node instanceof AST_Finally) {
                level++;
            } else if (node instanceof AST_IterationStatement) {
                in_loop = true;
            } else if (node instanceof AST_Scope) {
                scope = node;
                break;
            } else if (node instanceof AST_Try) {
                in_try = true;
            }
        } while (node = compressor.parent(level++));
    }

    // Search from right to left for assignment-like expressions:
    // - `var a = x;`
    // - `a = x;`
    // - `++a`
    // For each candidate, scan from left to right for first usage, then try
    // to fold assignment into the site for compression.
    // Will not attempt to collapse assignments into or past code blocks
    // which are not sequentially executed, e.g. loops and conditionals.
    function collapse(statements, compressor) {
        if (scope.pinned()) return statements;
        var args;
        var candidates = [];
        var stat_index = statements.length;
        var scanner = new TreeTransformer(function(node) {
            if (abort) return node;
            // Skip nodes before `candidate` as quickly as possible
            if (!hit) {
                if (node !== hit_stack[hit_index]) return node;
                hit_index++;
                if (hit_index < hit_stack.length) return handle_custom_scan_order(node);
                hit = true;
                stop_after = find_stop(node, 0);
                if (stop_after === node) abort = true;
                return node;
            }
            // Stop immediately if these node types are encountered
            var parent = scanner.parent();
            if (node instanceof AST_Assign
                    && (node.logical || node.operator != "=" && lhs.equivalent_to(node.left))
                || node instanceof AST_Await
                || node instanceof AST_Call && lhs instanceof AST_PropAccess && lhs.equivalent_to(node.expression)
                || node instanceof AST_Debugger
                || node instanceof AST_Destructuring
                || node instanceof AST_Expansion
                    && node.expression instanceof AST_Symbol
                    && (
                        node.expression instanceof AST_This
                        || node.expression.definition().references.length > 1
                    )
                || node instanceof AST_IterationStatement && !(node instanceof AST_For)
                || node instanceof AST_LoopControl
                || node instanceof AST_Try
                || node instanceof AST_With
                || node instanceof AST_Yield
                || node instanceof AST_Export
                || node instanceof AST_Class
                || parent instanceof AST_For && node !== parent.init
                || !replace_all
                    && (
                        node instanceof AST_SymbolRef
                        && !node.is_declared(compressor)
                        && !pure_prop_access_globals.has(node))
                || node instanceof AST_SymbolRef
                    && parent instanceof AST_Call
                    && has_annotation(parent, _NOINLINE)
            ) {
                abort = true;
                return node;
            }
            // Stop only if candidate is found within conditional branches
            if (!stop_if_hit && (!lhs_local || !replace_all)
                && (parent instanceof AST_Binary && lazy_op.has(parent.operator) && parent.left !== node
                    || parent instanceof AST_Conditional && parent.condition !== node
                    || parent instanceof AST_If && parent.condition !== node)) {
                stop_if_hit = parent;
            }
            // Replace variable with assignment when found
            if (can_replace
                && !(node instanceof AST_SymbolDeclaration)
                && lhs.equivalent_to(node)
            ) {
                if (stop_if_hit) {
                    abort = true;
                    return node;
                }
                if (is_lhs(node, parent)) {
                    if (value_def) replaced++;
                    return node;
                } else {
                    replaced++;
                    if (value_def && candidate instanceof AST_VarDef) return node;
                }
                CHANGED = abort = true;
                if (candidate instanceof AST_UnaryPostfix) {
                    return make_node(AST_UnaryPrefix, candidate, candidate);
                }
                if (candidate instanceof AST_VarDef) {
                    var def = candidate.name.definition();
                    var value = candidate.value;
                    if (def.references.length - def.replaced == 1 && !compressor.exposed(def)) {
                        def.replaced++;
                        if (funarg && is_identifier_atom(value)) {
                            return value.transform(compressor);
                        } else {
                            return maintain_this_binding(parent, node, value);
                        }
                    }
                    return make_node(AST_Assign, candidate, {
                        operator: "=",
                        logical: false,
                        left: make_node(AST_SymbolRef, candidate.name, candidate.name),
                        right: value
                    });
                }
                clear_flag(candidate, WRITE_ONLY);
                return candidate;
            }
            // These node types have child nodes that execute sequentially,
            // but are otherwise not safe to scan into or beyond them.
            var sym;
            if (node instanceof AST_Call
                || node instanceof AST_Exit
                    && (side_effects || lhs instanceof AST_PropAccess || may_modify(lhs))
                || node instanceof AST_PropAccess
                    && (side_effects || node.expression.may_throw_on_access(compressor))
                || node instanceof AST_SymbolRef
                    && (lvalues.get(node.name) || side_effects && may_modify(node))
                || node instanceof AST_VarDef && node.value
                    && (lvalues.has(node.name.name) || side_effects && may_modify(node.name))
                || (sym = is_lhs(node.left, node))
                    && (sym instanceof AST_PropAccess || lvalues.has(sym.name))
                || may_throw
                    && (in_try ? node.has_side_effects(compressor) : side_effects_external(node))) {
                stop_after = node;
                if (node instanceof AST_Scope) abort = true;
            }
            return handle_custom_scan_order(node);
        }, function(node) {
            if (abort) return;
            if (stop_after === node) abort = true;
            if (stop_if_hit === node) stop_if_hit = null;
        });
        var multi_replacer = new TreeTransformer(function(node) {
            if (abort) return node;
            // Skip nodes before `candidate` as quickly as possible
            if (!hit) {
                if (node !== hit_stack[hit_index]) return node;
                hit_index++;
                if (hit_index < hit_stack.length) return;
                hit = true;
                return node;
            }
            // Replace variable when found
            if (node instanceof AST_SymbolRef
                && node.name == def.name) {
                if (!--replaced) abort = true;
                if (is_lhs(node, multi_replacer.parent())) return node;
                def.replaced++;
                value_def.replaced--;
                return candidate.value;
            }
            // Skip (non-executed) functions and (leading) default case in switch statements
            if (node instanceof AST_Default || node instanceof AST_Scope) return node;
        });
        while (--stat_index >= 0) {
            // Treat parameters as collapsible in IIFE, i.e.
            //   function(a, b){ ... }(x());
            // would be translated into equivalent assignments:
            //   var a = x(), b = undefined;
            if (stat_index == 0 && compressor.option("unused")) extract_args();
            // Find collapsible assignments
            var hit_stack = [];
            extract_candidates(statements[stat_index]);
            while (candidates.length > 0) {
                hit_stack = candidates.pop();
                var hit_index = 0;
                var candidate = hit_stack[hit_stack.length - 1];
                var value_def = null;
                var stop_after = null;
                var stop_if_hit = null;
                var lhs = get_lhs(candidate);
                if (!lhs || is_lhs_read_only(lhs) || lhs.has_side_effects(compressor)) continue;
                // Locate symbols which may execute code outside of scanning range
                var lvalues = get_lvalues(candidate);
                var lhs_local = is_lhs_local(lhs);
                if (lhs instanceof AST_SymbolRef) lvalues.set(lhs.name, false);
                var side_effects = value_has_side_effects(candidate);
                var replace_all = replace_all_symbols();
                var may_throw = candidate.may_throw(compressor);
                var funarg = candidate.name instanceof AST_SymbolFunarg;
                var hit = funarg;
                var abort = false, replaced = 0, can_replace = !args || !hit;
                if (!can_replace) {
                    for (var j = compressor.self().argnames.lastIndexOf(candidate.name) + 1; !abort && j < args.length; j++) {
                        args[j].transform(scanner);
                    }
                    can_replace = true;
                }
                for (var i = stat_index; !abort && i < statements.length; i++) {
                    statements[i].transform(scanner);
                }
                if (value_def) {
                    var def = candidate.name.definition();
                    if (abort && def.references.length - def.replaced > replaced) replaced = false;
                    else {
                        abort = false;
                        hit_index = 0;
                        hit = funarg;
                        for (var i = stat_index; !abort && i < statements.length; i++) {
                            statements[i].transform(multi_replacer);
                        }
                        value_def.single_use = false;
                    }
                }
                if (replaced && !remove_candidate(candidate)) statements.splice(stat_index, 1);
            }
        }

        function handle_custom_scan_order(node) {
            // Skip (non-executed) functions
            if (node instanceof AST_Scope) return node;

            // Scan case expressions first in a switch statement
            if (node instanceof AST_Switch) {
                node.expression = node.expression.transform(scanner);
                for (var i = 0, len = node.body.length; !abort && i < len; i++) {
                    var branch = node.body[i];
                    if (branch instanceof AST_Case) {
                        if (!hit) {
                            if (branch !== hit_stack[hit_index]) continue;
                            hit_index++;
                        }
                        branch.expression = branch.expression.transform(scanner);
                        if (!replace_all) break;
                    }
                }
                abort = true;
                return node;
            }
        }

        function redefined_within_scope(def, scope) {
            if (def.global) return false;
            let cur_scope = def.scope;
            while (cur_scope && cur_scope !== scope) {
                if (cur_scope.variables.has(def.name)) return true;
                cur_scope = cur_scope.parent_scope;
            }
            return false;
        }

        function has_overlapping_symbol(fn, arg, fn_strict) {
            var found = false, scan_this = !(fn instanceof AST_Arrow);
            arg.walk(new TreeWalker(function(node, descend) {
                if (found) return true;
                if (node instanceof AST_SymbolRef && (fn.variables.has(node.name) || redefined_within_scope(node.definition(), fn))) {
                    var s = node.definition().scope;
                    if (s !== scope) while (s = s.parent_scope) {
                        if (s === scope) return true;
                    }
                    return found = true;
                }
                if ((fn_strict || scan_this) && node instanceof AST_This) {
                    return found = true;
                }
                if (node instanceof AST_Scope && !(node instanceof AST_Arrow)) {
                    var prev = scan_this;
                    scan_this = false;
                    descend();
                    scan_this = prev;
                    return true;
                }
            }));
            return found;
        }

        function extract_args() {
            var iife, fn = compressor.self();
            if (is_func_expr(fn)
                && !fn.name
                && !fn.uses_arguments
                && !fn.pinned()
                && (iife = compressor.parent()) instanceof AST_Call
                && iife.expression === fn
                && iife.args.every((arg) => !(arg instanceof AST_Expansion))
            ) {
                var fn_strict = compressor.has_directive("use strict");
                if (fn_strict && !member(fn_strict, fn.body)) fn_strict = false;
                var len = fn.argnames.length;
                args = iife.args.slice(len);
                var names = new Set();
                for (var i = len; --i >= 0;) {
                    var sym = fn.argnames[i];
                    var arg = iife.args[i];
                    // The following two line fix is a duplicate of the fix at
                    // https://github.com/terser/terser/commit/011d3eb08cefe6922c7d1bdfa113fc4aeaca1b75
                    // This might mean that these two pieces of code (one here in collapse_vars and another in reduce_vars
                    // Might be doing the exact same thing.
                    const def = sym.definition && sym.definition();
                    const is_reassigned = def && def.orig.length > 1;
                    if (is_reassigned) continue;
                    args.unshift(make_node(AST_VarDef, sym, {
                        name: sym,
                        value: arg
                    }));
                    if (names.has(sym.name)) continue;
                    names.add(sym.name);
                    if (sym instanceof AST_Expansion) {
                        var elements = iife.args.slice(i);
                        if (elements.every((arg) =>
                            !has_overlapping_symbol(fn, arg, fn_strict)
                        )) {
                            candidates.unshift([ make_node(AST_VarDef, sym, {
                                name: sym.expression,
                                value: make_node(AST_Array, iife, {
                                    elements: elements
                                })
                            }) ]);
                        }
                    } else {
                        if (!arg) {
                            arg = make_node(AST_Undefined, sym).transform(compressor);
                        } else if (arg instanceof AST_Lambda && arg.pinned()
                            || has_overlapping_symbol(fn, arg, fn_strict)
                        ) {
                            arg = null;
                        }
                        if (arg) candidates.unshift([ make_node(AST_VarDef, sym, {
                            name: sym,
                            value: arg
                        }) ]);
                    }
                }
            }
        }

        function extract_candidates(expr) {
            hit_stack.push(expr);
            if (expr instanceof AST_Assign) {
                if (!expr.left.has_side_effects(compressor)) {
                    candidates.push(hit_stack.slice());
                }
                extract_candidates(expr.right);
            } else if (expr instanceof AST_Binary) {
                extract_candidates(expr.left);
                extract_candidates(expr.right);
            } else if (expr instanceof AST_Call && !has_annotation(expr, _NOINLINE)) {
                extract_candidates(expr.expression);
                expr.args.forEach(extract_candidates);
            } else if (expr instanceof AST_Case) {
                extract_candidates(expr.expression);
            } else if (expr instanceof AST_Conditional) {
                extract_candidates(expr.condition);
                extract_candidates(expr.consequent);
                extract_candidates(expr.alternative);
            } else if (expr instanceof AST_Definitions) {
                var len = expr.definitions.length;
                // limit number of trailing variable definitions for consideration
                var i = len - 200;
                if (i < 0) i = 0;
                for (; i < len; i++) {
                    extract_candidates(expr.definitions[i]);
                }
            } else if (expr instanceof AST_DWLoop) {
                extract_candidates(expr.condition);
                if (!(expr.body instanceof AST_Block)) {
                    extract_candidates(expr.body);
                }
            } else if (expr instanceof AST_Exit) {
                if (expr.value) extract_candidates(expr.value);
            } else if (expr instanceof AST_For) {
                if (expr.init) extract_candidates(expr.init);
                if (expr.condition) extract_candidates(expr.condition);
                if (expr.step) extract_candidates(expr.step);
                if (!(expr.body instanceof AST_Block)) {
                    extract_candidates(expr.body);
                }
            } else if (expr instanceof AST_ForIn) {
                extract_candidates(expr.object);
                if (!(expr.body instanceof AST_Block)) {
                    extract_candidates(expr.body);
                }
            } else if (expr instanceof AST_If) {
                extract_candidates(expr.condition);
                if (!(expr.body instanceof AST_Block)) {
                    extract_candidates(expr.body);
                }
                if (expr.alternative && !(expr.alternative instanceof AST_Block)) {
                    extract_candidates(expr.alternative);
                }
            } else if (expr instanceof AST_Sequence) {
                expr.expressions.forEach(extract_candidates);
            } else if (expr instanceof AST_SimpleStatement) {
                extract_candidates(expr.body);
            } else if (expr instanceof AST_Switch) {
                extract_candidates(expr.expression);
                expr.body.forEach(extract_candidates);
            } else if (expr instanceof AST_Unary) {
                if (expr.operator == "++" || expr.operator == "--") {
                    candidates.push(hit_stack.slice());
                }
            } else if (expr instanceof AST_VarDef) {
                if (expr.value) {
                    candidates.push(hit_stack.slice());
                    extract_candidates(expr.value);
                }
            }
            hit_stack.pop();
        }

        function find_stop(node, level, write_only) {
            var parent = scanner.parent(level);
            if (parent instanceof AST_Assign) {
                if (write_only
                    && !parent.logical
                    && !(parent.left instanceof AST_PropAccess
                        || lvalues.has(parent.left.name))) {
                    return find_stop(parent, level + 1, write_only);
                }
                return node;
            }
            if (parent instanceof AST_Binary) {
                if (write_only && (!lazy_op.has(parent.operator) || parent.left === node)) {
                    return find_stop(parent, level + 1, write_only);
                }
                return node;
            }
            if (parent instanceof AST_Call) return node;
            if (parent instanceof AST_Case) return node;
            if (parent instanceof AST_Conditional) {
                if (write_only && parent.condition === node) {
                    return find_stop(parent, level + 1, write_only);
                }
                return node;
            }
            if (parent instanceof AST_Definitions) {
                return find_stop(parent, level + 1, true);
            }
            if (parent instanceof AST_Exit) {
                return write_only ? find_stop(parent, level + 1, write_only) : node;
            }
            if (parent instanceof AST_If) {
                if (write_only && parent.condition === node) {
                    return find_stop(parent, level + 1, write_only);
                }
                return node;
            }
            if (parent instanceof AST_IterationStatement) return node;
            if (parent instanceof AST_Sequence) {
                return find_stop(parent, level + 1, parent.tail_node() !== node);
            }
            if (parent instanceof AST_SimpleStatement) {
                return find_stop(parent, level + 1, true);
            }
            if (parent instanceof AST_Switch) return node;
            if (parent instanceof AST_VarDef) return node;
            return null;
        }

        function mangleable_var(var_def) {
            var value = var_def.value;
            if (!(value instanceof AST_SymbolRef)) return;
            if (value.name == "arguments") return;
            var def = value.definition();
            if (def.undeclared) return;
            return value_def = def;
        }

        function get_lhs(expr) {
            if (expr instanceof AST_Assign && expr.logical) {
                return false;
            } else if (expr instanceof AST_VarDef && expr.name instanceof AST_SymbolDeclaration) {
                var def = expr.name.definition();
                if (!member(expr.name, def.orig)) return;
                var referenced = def.references.length - def.replaced;
                if (!referenced) return;
                var declared = def.orig.length - def.eliminated;
                if (declared > 1 && !(expr.name instanceof AST_SymbolFunarg)
                    || (referenced > 1 ? mangleable_var(expr) : !compressor.exposed(def))) {
                    return make_node(AST_SymbolRef, expr.name, expr.name);
                }
            } else {
                const lhs = expr instanceof AST_Assign
                    ? expr.left
                    : expr.expression;
                return !is_ref_of(lhs, AST_SymbolConst)
                    && !is_ref_of(lhs, AST_SymbolLet) && lhs;
            }
        }

        function get_rvalue(expr) {
            if (expr instanceof AST_Assign) {
                return expr.right;
            } else {
                return expr.value;
            }
        }

        function get_lvalues(expr) {
            var lvalues = new Map();
            if (expr instanceof AST_Unary) return lvalues;
            var tw = new TreeWalker(function(node) {
                var sym = node;
                while (sym instanceof AST_PropAccess) sym = sym.expression;
                if (sym instanceof AST_SymbolRef || sym instanceof AST_This) {
                    lvalues.set(sym.name, lvalues.get(sym.name) || is_modified(compressor, tw, node, node, 0));
                }
            });
            get_rvalue(expr).walk(tw);
            return lvalues;
        }

        function remove_candidate(expr) {
            if (expr.name instanceof AST_SymbolFunarg) {
                var iife = compressor.parent(), argnames = compressor.self().argnames;
                var index = argnames.indexOf(expr.name);
                if (index < 0) {
                    iife.args.length = Math.min(iife.args.length, argnames.length - 1);
                } else {
                    var args = iife.args;
                    if (args[index]) args[index] = make_node(AST_Number, args[index], {
                        value: 0
                    });
                }
                return true;
            }
            var found = false;
            return statements[stat_index].transform(new TreeTransformer(function(node, descend, in_list) {
                if (found) return node;
                if (node === expr || node.body === expr) {
                    found = true;
                    if (node instanceof AST_VarDef) {
                        node.value = node.name instanceof AST_SymbolConst
                             ? make_node(AST_Undefined, node.value)  // `const` always needs value.
                             : null;
                        return node;
                    }
                    return in_list ? MAP.skip : null;
                }
            }, function(node) {
                if (node instanceof AST_Sequence) switch (node.expressions.length) {
                  case 0: return null;
                  case 1: return node.expressions[0];
                }
            }));
        }

        function is_lhs_local(lhs) {
            while (lhs instanceof AST_PropAccess) lhs = lhs.expression;
            return lhs instanceof AST_SymbolRef
                && lhs.definition().scope === scope
                && !(in_loop
                    && (lvalues.has(lhs.name)
                        || candidate instanceof AST_Unary
                        || (candidate instanceof AST_Assign
                            && !candidate.logical
                            && candidate.operator != "=")));
        }

        function value_has_side_effects(expr) {
            if (expr instanceof AST_Unary) return unary_side_effects.has(expr.operator);
            return get_rvalue(expr).has_side_effects(compressor);
        }

        function replace_all_symbols() {
            if (side_effects) return false;
            if (value_def) return true;
            if (lhs instanceof AST_SymbolRef) {
                var def = lhs.definition();
                if (def.references.length - def.replaced == (candidate instanceof AST_VarDef ? 1 : 2)) {
                    return true;
                }
            }
            return false;
        }

        function may_modify(sym) {
            if (!sym.definition) return true; // AST_Destructuring
            var def = sym.definition();
            if (def.orig.length == 1 && def.orig[0] instanceof AST_SymbolDefun) return false;
            if (def.scope.get_defun_scope() !== scope) return true;
            return !def.references.every((ref) => {
                var s = ref.scope.get_defun_scope();
                // "block" scope within AST_Catch
                if (s.TYPE == "Scope") s = s.parent_scope;
                return s === scope;
            });
        }

        function side_effects_external(node, lhs) {
            if (node instanceof AST_Assign) return side_effects_external(node.left, true);
            if (node instanceof AST_Unary) return side_effects_external(node.expression, true);
            if (node instanceof AST_VarDef) return node.value && side_effects_external(node.value);
            if (lhs) {
                if (node instanceof AST_Dot) return side_effects_external(node.expression, true);
                if (node instanceof AST_Sub) return side_effects_external(node.expression, true);
                if (node instanceof AST_SymbolRef) return node.definition().scope !== scope;
            }
            return false;
        }
    }

    function eliminate_spurious_blocks(statements) {
        var seen_dirs = [];
        for (var i = 0; i < statements.length;) {
            var stat = statements[i];
            if (stat instanceof AST_BlockStatement && stat.body.every(can_be_evicted_from_block)) {
                CHANGED = true;
                eliminate_spurious_blocks(stat.body);
                statements.splice(i, 1, ...stat.body);
                i += stat.body.length;
            } else if (stat instanceof AST_EmptyStatement) {
                CHANGED = true;
                statements.splice(i, 1);
            } else if (stat instanceof AST_Directive) {
                if (seen_dirs.indexOf(stat.value) < 0) {
                    i++;
                    seen_dirs.push(stat.value);
                } else {
                    CHANGED = true;
                    statements.splice(i, 1);
                }
            } else i++;
        }
    }

    function handle_if_return(statements, compressor) {
        var self = compressor.self();
        var multiple_if_returns = has_multiple_if_returns(statements);
        var in_lambda = self instanceof AST_Lambda;
        for (var i = statements.length; --i >= 0;) {
            var stat = statements[i];
            var j = next_index(i);
            var next = statements[j];

            if (in_lambda && !next && stat instanceof AST_Return) {
                if (!stat.value) {
                    CHANGED = true;
                    statements.splice(i, 1);
                    continue;
                }
                if (stat.value instanceof AST_UnaryPrefix && stat.value.operator == "void") {
                    CHANGED = true;
                    statements[i] = make_node(AST_SimpleStatement, stat, {
                        body: stat.value.expression
                    });
                    continue;
                }
            }

            if (stat instanceof AST_If) {
                var ab = aborts(stat.body);
                if (can_merge_flow(ab)) {
                    if (ab.label) {
                        remove(ab.label.thedef.references, ab);
                    }
                    CHANGED = true;
                    stat = stat.clone();
                    stat.condition = stat.condition.negate(compressor);
                    var body = as_statement_array_with_return(stat.body, ab);
                    stat.body = make_node(AST_BlockStatement, stat, {
                        body: as_statement_array(stat.alternative).concat(extract_functions())
                    });
                    stat.alternative = make_node(AST_BlockStatement, stat, {
                        body: body
                    });
                    statements[i] = stat.transform(compressor);
                    continue;
                }

                var ab = aborts(stat.alternative);
                if (can_merge_flow(ab)) {
                    if (ab.label) {
                        remove(ab.label.thedef.references, ab);
                    }
                    CHANGED = true;
                    stat = stat.clone();
                    stat.body = make_node(AST_BlockStatement, stat.body, {
                        body: as_statement_array(stat.body).concat(extract_functions())
                    });
                    var body = as_statement_array_with_return(stat.alternative, ab);
                    stat.alternative = make_node(AST_BlockStatement, stat.alternative, {
                        body: body
                    });
                    statements[i] = stat.transform(compressor);
                    continue;
                }
            }

            if (stat instanceof AST_If && stat.body instanceof AST_Return) {
                var value = stat.body.value;
                //---
                // pretty silly case, but:
                // if (foo()) return; return; ==> foo(); return;
                if (!value && !stat.alternative
                    && (in_lambda && !next || next instanceof AST_Return && !next.value)) {
                    CHANGED = true;
                    statements[i] = make_node(AST_SimpleStatement, stat.condition, {
                        body: stat.condition
                    });
                    continue;
                }
                //---
                // if (foo()) return x; return y; ==> return foo() ? x : y;
                if (value && !stat.alternative && next instanceof AST_Return && next.value) {
                    CHANGED = true;
                    stat = stat.clone();
                    stat.alternative = next;
                    statements[i] = stat.transform(compressor);
                    statements.splice(j, 1);
                    continue;
                }
                //---
                // if (foo()) return x; [ return ; ] ==> return foo() ? x : undefined;
                if (value && !stat.alternative
                    && (!next && in_lambda && multiple_if_returns
                        || next instanceof AST_Return)) {
                    CHANGED = true;
                    stat = stat.clone();
                    stat.alternative = next || make_node(AST_Return, stat, {
                        value: null
                    });
                    statements[i] = stat.transform(compressor);
                    if (next) statements.splice(j, 1);
                    continue;
                }
                //---
                // if (a) return b; if (c) return d; e; ==> return a ? b : c ? d : void e;
                //
                // if sequences is not enabled, this can lead to an endless loop (issue #866).
                // however, with sequences on this helps producing slightly better output for
                // the example code.
                var prev = statements[prev_index(i)];
                if (compressor.option("sequences") && in_lambda && !stat.alternative
                    && prev instanceof AST_If && prev.body instanceof AST_Return
                    && next_index(j) == statements.length && next instanceof AST_SimpleStatement) {
                    CHANGED = true;
                    stat = stat.clone();
                    stat.alternative = make_node(AST_BlockStatement, next, {
                        body: [
                            next,
                            make_node(AST_Return, next, {
                                value: null
                            })
                        ]
                    });
                    statements[i] = stat.transform(compressor);
                    statements.splice(j, 1);
                    continue;
                }
            }
        }

        function has_multiple_if_returns(statements) {
            var n = 0;
            for (var i = statements.length; --i >= 0;) {
                var stat = statements[i];
                if (stat instanceof AST_If && stat.body instanceof AST_Return) {
                    if (++n > 1) return true;
                }
            }
            return false;
        }

        function is_return_void(value) {
            return !value || value instanceof AST_UnaryPrefix && value.operator == "void";
        }

        function can_merge_flow(ab) {
            if (!ab) return false;
            for (var j = i + 1, len = statements.length; j < len; j++) {
                var stat = statements[j];
                if (stat instanceof AST_Const || stat instanceof AST_Let) return false;
            }
            var lct = ab instanceof AST_LoopControl ? compressor.loopcontrol_target(ab) : null;
            return ab instanceof AST_Return && in_lambda && is_return_void(ab.value)
                || ab instanceof AST_Continue && self === loop_body(lct)
                || ab instanceof AST_Break && lct instanceof AST_BlockStatement && self === lct;
        }

        function extract_functions() {
            var tail = statements.slice(i + 1);
            statements.length = i + 1;
            return tail.filter(function(stat) {
                if (stat instanceof AST_Defun) {
                    statements.push(stat);
                    return false;
                }
                return true;
            });
        }

        function as_statement_array_with_return(node, ab) {
            var body = as_statement_array(node).slice(0, -1);
            if (ab.value) {
                body.push(make_node(AST_SimpleStatement, ab.value, {
                    body: ab.value.expression
                }));
            }
            return body;
        }

        function next_index(i) {
            for (var j = i + 1, len = statements.length; j < len; j++) {
                var stat = statements[j];
                if (!(stat instanceof AST_Var && declarations_only(stat))) {
                    break;
                }
            }
            return j;
        }

        function prev_index(i) {
            for (var j = i; --j >= 0;) {
                var stat = statements[j];
                if (!(stat instanceof AST_Var && declarations_only(stat))) {
                    break;
                }
            }
            return j;
        }
    }

    function eliminate_dead_code(statements, compressor) {
        var has_quit;
        var self = compressor.self();
        for (var i = 0, n = 0, len = statements.length; i < len; i++) {
            var stat = statements[i];
            if (stat instanceof AST_LoopControl) {
                var lct = compressor.loopcontrol_target(stat);
                if (stat instanceof AST_Break
                        && !(lct instanceof AST_IterationStatement)
                        && loop_body(lct) === self
                    || stat instanceof AST_Continue
                        && loop_body(lct) === self) {
                    if (stat.label) {
                        remove(stat.label.thedef.references, stat);
                    }
                } else {
                    statements[n++] = stat;
                }
            } else {
                statements[n++] = stat;
            }
            if (aborts(stat)) {
                has_quit = statements.slice(i + 1);
                break;
            }
        }
        statements.length = n;
        CHANGED = n != len;
        if (has_quit) has_quit.forEach(function(stat) {
            trim_unreachable_code(compressor, stat, statements);
        });
    }

    function declarations_only(node) {
        return node.definitions.every((var_def) =>
            !var_def.value
        );
    }

    function sequencesize(statements, compressor) {
        if (statements.length < 2) return;
        var seq = [], n = 0;
        function push_seq() {
            if (!seq.length) return;
            var body = make_sequence(seq[0], seq);
            statements[n++] = make_node(AST_SimpleStatement, body, { body: body });
            seq = [];
        }
        for (var i = 0, len = statements.length; i < len; i++) {
            var stat = statements[i];
            if (stat instanceof AST_SimpleStatement) {
                if (seq.length >= compressor.sequences_limit) push_seq();
                var body = stat.body;
                if (seq.length > 0) body = body.drop_side_effect_free(compressor);
                if (body) merge_sequence(seq, body);
            } else if (stat instanceof AST_Definitions && declarations_only(stat)
                || stat instanceof AST_Defun) {
                statements[n++] = stat;
            } else {
                push_seq();
                statements[n++] = stat;
            }
        }
        push_seq();
        statements.length = n;
        if (n != len) CHANGED = true;
    }

    function to_simple_statement(block, decls) {
        if (!(block instanceof AST_BlockStatement)) return block;
        var stat = null;
        for (var i = 0, len = block.body.length; i < len; i++) {
            var line = block.body[i];
            if (line instanceof AST_Var && declarations_only(line)) {
                decls.push(line);
            } else if (stat) {
                return false;
            } else {
                stat = line;
            }
        }
        return stat;
    }

    function sequencesize_2(statements, compressor) {
        function cons_seq(right) {
            n--;
            CHANGED = true;
            var left = prev.body;
            return make_sequence(left, [ left, right ]).transform(compressor);
        }
        var n = 0, prev;
        for (var i = 0; i < statements.length; i++) {
            var stat = statements[i];
            if (prev) {
                if (stat instanceof AST_Exit) {
                    stat.value = cons_seq(stat.value || make_node(AST_Undefined, stat).transform(compressor));
                } else if (stat instanceof AST_For) {
                    if (!(stat.init instanceof AST_Definitions)) {
                        const abort = walk(prev.body, node => {
                            if (node instanceof AST_Scope) return true;
                            if (
                                node instanceof AST_Binary
                                && node.operator === "in"
                            ) {
                                return walk_abort;
                            }
                        });
                        if (!abort) {
                            if (stat.init) stat.init = cons_seq(stat.init);
                            else {
                                stat.init = prev.body;
                                n--;
                                CHANGED = true;
                            }
                        }
                    }
                } else if (stat instanceof AST_ForIn) {
                    if (!(stat.init instanceof AST_Const) && !(stat.init instanceof AST_Let)) {
                        stat.object = cons_seq(stat.object);
                    }
                } else if (stat instanceof AST_If) {
                    stat.condition = cons_seq(stat.condition);
                } else if (stat instanceof AST_Switch) {
                    stat.expression = cons_seq(stat.expression);
                } else if (stat instanceof AST_With) {
                    stat.expression = cons_seq(stat.expression);
                }
            }
            if (compressor.option("conditionals") && stat instanceof AST_If) {
                var decls = [];
                var body = to_simple_statement(stat.body, decls);
                var alt = to_simple_statement(stat.alternative, decls);
                if (body !== false && alt !== false && decls.length > 0) {
                    var len = decls.length;
                    decls.push(make_node(AST_If, stat, {
                        condition: stat.condition,
                        body: body || make_node(AST_EmptyStatement, stat.body),
                        alternative: alt
                    }));
                    decls.unshift(n, 1);
                    [].splice.apply(statements, decls);
                    i += len;
                    n += len + 1;
                    prev = null;
                    CHANGED = true;
                    continue;
                }
            }
            statements[n++] = stat;
            prev = stat instanceof AST_SimpleStatement ? stat : null;
        }
        statements.length = n;
    }

    function join_object_assignments(defn, body) {
        if (!(defn instanceof AST_Definitions)) return;
        var def = defn.definitions[defn.definitions.length - 1];
        if (!(def.value instanceof AST_Object)) return;
        var exprs;
        if (body instanceof AST_Assign && !body.logical) {
            exprs = [ body ];
        } else if (body instanceof AST_Sequence) {
            exprs = body.expressions.slice();
        }
        if (!exprs) return;
        var trimmed = false;
        do {
            var node = exprs[0];
            if (!(node instanceof AST_Assign)) break;
            if (node.operator != "=") break;
            if (!(node.left instanceof AST_PropAccess)) break;
            var sym = node.left.expression;
            if (!(sym instanceof AST_SymbolRef)) break;
            if (def.name.name != sym.name) break;
            if (!node.right.is_constant_expression(scope)) break;
            var prop = node.left.property;
            if (prop instanceof AST_Node) {
                prop = prop.evaluate(compressor);
            }
            if (prop instanceof AST_Node) break;
            prop = "" + prop;
            var diff = compressor.option("ecma") < 2015
                && compressor.has_directive("use strict") ? function(node) {
                return node.key != prop && (node.key && node.key.name != prop);
            } : function(node) {
                return node.key && node.key.name != prop;
            };
            if (!def.value.properties.every(diff)) break;
            var p = def.value.properties.filter(function (p) { return p.key === prop; })[0];
            if (!p) {
                def.value.properties.push(make_node(AST_ObjectKeyVal, node, {
                    key: prop,
                    value: node.right
                }));
            } else {
                p.value = new AST_Sequence({
                    start: p.start,
                    expressions: [p.value.clone(), node.right.clone()],
                    end: p.end
                });
            }
            exprs.shift();
            trimmed = true;
        } while (exprs.length);
        return trimmed && exprs;
    }

    function join_consecutive_vars(statements) {
        var defs;
        for (var i = 0, j = -1, len = statements.length; i < len; i++) {
            var stat = statements[i];
            var prev = statements[j];
            if (stat instanceof AST_Definitions) {
                if (prev && prev.TYPE == stat.TYPE) {
                    prev.definitions = prev.definitions.concat(stat.definitions);
                    CHANGED = true;
                } else if (defs && defs.TYPE == stat.TYPE && declarations_only(stat)) {
                    defs.definitions = defs.definitions.concat(stat.definitions);
                    CHANGED = true;
                } else {
                    statements[++j] = stat;
                    defs = stat;
                }
            } else if (stat instanceof AST_Exit) {
                stat.value = extract_object_assignments(stat.value);
            } else if (stat instanceof AST_For) {
                var exprs = join_object_assignments(prev, stat.init);
                if (exprs) {
                    CHANGED = true;
                    stat.init = exprs.length ? make_sequence(stat.init, exprs) : null;
                    statements[++j] = stat;
                } else if (prev instanceof AST_Var && (!stat.init || stat.init.TYPE == prev.TYPE)) {
                    if (stat.init) {
                        prev.definitions = prev.definitions.concat(stat.init.definitions);
                    }
                    stat.init = prev;
                    statements[j] = stat;
                    CHANGED = true;
                } else if (defs && stat.init && defs.TYPE == stat.init.TYPE && declarations_only(stat.init)) {
                    defs.definitions = defs.definitions.concat(stat.init.definitions);
                    stat.init = null;
                    statements[++j] = stat;
                    CHANGED = true;
                } else {
                    statements[++j] = stat;
                }
            } else if (stat instanceof AST_ForIn) {
                stat.object = extract_object_assignments(stat.object);
            } else if (stat instanceof AST_If) {
                stat.condition = extract_object_assignments(stat.condition);
            } else if (stat instanceof AST_SimpleStatement) {
                var exprs = join_object_assignments(prev, stat.body);
                if (exprs) {
                    CHANGED = true;
                    if (!exprs.length) continue;
                    stat.body = make_sequence(stat.body, exprs);
                }
                statements[++j] = stat;
            } else if (stat instanceof AST_Switch) {
                stat.expression = extract_object_assignments(stat.expression);
            } else if (stat instanceof AST_With) {
                stat.expression = extract_object_assignments(stat.expression);
            } else {
                statements[++j] = stat;
            }
        }
        statements.length = j + 1;

        function extract_object_assignments(value) {
            statements[++j] = stat;
            var exprs = join_object_assignments(prev, value);
            if (exprs) {
                CHANGED = true;
                if (exprs.length) {
                    return make_sequence(value, exprs);
                } else if (value instanceof AST_Sequence) {
                    return value.tail_node().left;
                } else {
                    return value.left;
                }
            }
            return value;
        }
    }
}

function trim_unreachable_code(compressor, stat, target) {
    walk(stat, node => {
        if (node instanceof AST_Var) {
            node.remove_initializers();
            target.push(node);
            return true;
        }
        if (
            node instanceof AST_Defun
            && (node === stat || !compressor.has_directive("use strict"))
        ) {
            target.push(node === stat ? node : make_node(AST_Var, node, {
                definitions: [
                    make_node(AST_VarDef, node, {
                        name: make_node(AST_SymbolVar, node.name, node.name),
                        value: null
                    })
                ]
            }));
            return true;
        }
        if (node instanceof AST_Export || node instanceof AST_Import) {
            target.push(node);
            return true;
        }
        if (node instanceof AST_Scope) {
            return true;
        }
    });
}

function get_value(key) {
    if (key instanceof AST_Constant) {
        return key.getValue();
    }
    if (key instanceof AST_UnaryPrefix
        && key.operator == "void"
        && key.expression instanceof AST_Constant) {
        return;
    }
    return key;
}

function is_undefined(node, compressor) {
    return (
        has_flag(node, UNDEFINED)
        || node instanceof AST_Undefined
        || node instanceof AST_UnaryPrefix
            && node.operator == "void"
            && !node.expression.has_side_effects(compressor)
    );
}

// may_throw_on_access()
// returns true if this node may be null, undefined or contain `AST_Accessor`
(function(def_may_throw_on_access) {
    AST_Node.DEFMETHOD("may_throw_on_access", function(compressor) {
        return !compressor.option("pure_getters")
            || this._dot_throw(compressor);
    });

    function is_strict(compressor) {
        return /strict/.test(compressor.option("pure_getters"));
    }

    def_may_throw_on_access(AST_Node, is_strict);
    def_may_throw_on_access(AST_Null, return_true);
    def_may_throw_on_access(AST_Undefined, return_true);
    def_may_throw_on_access(AST_Constant, return_false);
    def_may_throw_on_access(AST_Array, return_false);
    def_may_throw_on_access(AST_Object, function(compressor) {
        if (!is_strict(compressor)) return false;
        for (var i = this.properties.length; --i >=0;)
            if (this.properties[i]._dot_throw(compressor)) return true;
        return false;
    });
    // Do not be as strict with classes as we are with objects.
    // Hopefully the community is not going to abuse static getters and setters.
    // https://github.com/terser/terser/issues/724#issuecomment-643655656
    def_may_throw_on_access(AST_Class, return_false);
    def_may_throw_on_access(AST_ObjectProperty, return_false);
    def_may_throw_on_access(AST_ObjectGetter, return_true);
    def_may_throw_on_access(AST_Expansion, function(compressor) {
        return this.expression._dot_throw(compressor);
    });
    def_may_throw_on_access(AST_Function, return_false);
    def_may_throw_on_access(AST_Arrow, return_false);
    def_may_throw_on_access(AST_UnaryPostfix, return_false);
    def_may_throw_on_access(AST_UnaryPrefix, function() {
        return this.operator == "void";
    });
    def_may_throw_on_access(AST_Binary, function(compressor) {
        return (this.operator == "&&" || this.operator == "||" || this.operator == "??")
            && (this.left._dot_throw(compressor) || this.right._dot_throw(compressor));
    });
    def_may_throw_on_access(AST_Assign, function(compressor) {
        if (this.logical) return true;

        return this.operator == "="
            && this.right._dot_throw(compressor);
    });
    def_may_throw_on_access(AST_Conditional, function(compressor) {
        return this.consequent._dot_throw(compressor)
            || this.alternative._dot_throw(compressor);
    });
    def_may_throw_on_access(AST_Dot, function(compressor) {
        if (!is_strict(compressor)) return false;

        if (this.property == "prototype") {
            return !(
                this.expression instanceof AST_Function
                || this.expression instanceof AST_Class
            );
        }
        return true;
    });
    def_may_throw_on_access(AST_Chain, function(compressor) {
        return this.expression._dot_throw(compressor);
    });
    def_may_throw_on_access(AST_Sequence, function(compressor) {
        return this.tail_node()._dot_throw(compressor);
    });
    def_may_throw_on_access(AST_SymbolRef, function(compressor) {
        if (this.name === "arguments") return false;
        if (has_flag(this, UNDEFINED)) return true;
        if (!is_strict(compressor)) return false;
        if (is_undeclared_ref(this) && this.is_declared(compressor)) return false;
        if (this.is_immutable()) return false;
        var fixed = this.fixed_value();
        return !fixed || fixed._dot_throw(compressor);
    });
})(function(node, func) {
    node.DEFMETHOD("_dot_throw", func);
});

/* -----[ boolean/negation helpers ]----- */

// methods to determine whether an expression has a boolean result type
(function(def_is_boolean) {
    const unary_bool = makePredicate("! delete");
    const binary_bool = makePredicate("in instanceof == != === !== < <= >= >");
    def_is_boolean(AST_Node, return_false);
    def_is_boolean(AST_UnaryPrefix, function() {
        return unary_bool.has(this.operator);
    });
    def_is_boolean(AST_Binary, function() {
        return binary_bool.has(this.operator)
            || lazy_op.has(this.operator)
                && this.left.is_boolean()
                && this.right.is_boolean();
    });
    def_is_boolean(AST_Conditional, function() {
        return this.consequent.is_boolean() && this.alternative.is_boolean();
    });
    def_is_boolean(AST_Assign, function() {
        return this.operator == "=" && this.right.is_boolean();
    });
    def_is_boolean(AST_Sequence, function() {
        return this.tail_node().is_boolean();
    });
    def_is_boolean(AST_True, return_true);
    def_is_boolean(AST_False, return_true);
})(function(node, func) {
    node.DEFMETHOD("is_boolean", func);
});

// methods to determine if an expression has a numeric result type
(function(def_is_number) {
    def_is_number(AST_Node, return_false);
    def_is_number(AST_Number, return_true);
    var unary = makePredicate("+ - ~ ++ --");
    def_is_number(AST_Unary, function() {
        return unary.has(this.operator);
    });
    var binary = makePredicate("- * / % & | ^ << >> >>>");
    def_is_number(AST_Binary, function(compressor) {
        return binary.has(this.operator) || this.operator == "+"
            && this.left.is_number(compressor)
            && this.right.is_number(compressor);
    });
    def_is_number(AST_Assign, function(compressor) {
        return binary.has(this.operator.slice(0, -1))
            || this.operator == "=" && this.right.is_number(compressor);
    });
    def_is_number(AST_Sequence, function(compressor) {
        return this.tail_node().is_number(compressor);
    });
    def_is_number(AST_Conditional, function(compressor) {
        return this.consequent.is_number(compressor) && this.alternative.is_number(compressor);
    });
})(function(node, func) {
    node.DEFMETHOD("is_number", func);
});

// methods to determine if an expression has a string result type
(function(def_is_string) {
    def_is_string(AST_Node, return_false);
    def_is_string(AST_String, return_true);
    def_is_string(AST_TemplateString, return_true);
    def_is_string(AST_UnaryPrefix, function() {
        return this.operator == "typeof";
    });
    def_is_string(AST_Binary, function(compressor) {
        return this.operator == "+" &&
            (this.left.is_string(compressor) || this.right.is_string(compressor));
    });
    def_is_string(AST_Assign, function(compressor) {
        return (this.operator == "=" || this.operator == "+=") && this.right.is_string(compressor);
    });
    def_is_string(AST_Sequence, function(compressor) {
        return this.tail_node().is_string(compressor);
    });
    def_is_string(AST_Conditional, function(compressor) {
        return this.consequent.is_string(compressor) && this.alternative.is_string(compressor);
    });
})(function(node, func) {
    node.DEFMETHOD("is_string", func);
});

var lazy_op = makePredicate("&& || ??");
var unary_side_effects = makePredicate("delete ++ --");

function is_lhs(node, parent) {
    if (parent instanceof AST_Unary && unary_side_effects.has(parent.operator)) return parent.expression;
    if (parent instanceof AST_Assign && parent.left === node) return node;
}

(function(def_find_defs) {
    function to_node(value, orig) {
        if (value instanceof AST_Node) return make_node(value.CTOR, orig, value);
        if (Array.isArray(value)) return make_node(AST_Array, orig, {
            elements: value.map(function(value) {
                return to_node(value, orig);
            })
        });
        if (value && typeof value == "object") {
            var props = [];
            for (var key in value) if (HOP(value, key)) {
                props.push(make_node(AST_ObjectKeyVal, orig, {
                    key: key,
                    value: to_node(value[key], orig)
                }));
            }
            return make_node(AST_Object, orig, {
                properties: props
            });
        }
        return make_node_from_constant(value, orig);
    }

    AST_Toplevel.DEFMETHOD("resolve_defines", function(compressor) {
        if (!compressor.option("global_defs")) return this;
        this.figure_out_scope({ ie8: compressor.option("ie8") });
        return this.transform(new TreeTransformer(function(node) {
            var def = node._find_defs(compressor, "");
            if (!def) return;
            var level = 0, child = node, parent;
            while (parent = this.parent(level++)) {
                if (!(parent instanceof AST_PropAccess)) break;
                if (parent.expression !== child) break;
                child = parent;
            }
            if (is_lhs(child, parent)) {
                return;
            }
            return def;
        }));
    });
    def_find_defs(AST_Node, noop);
    def_find_defs(AST_Chain, function(compressor, suffix) {
        return this.expression._find_defs(compressor, suffix);
    });
    def_find_defs(AST_Dot, function(compressor, suffix) {
        return this.expression._find_defs(compressor, "." + this.property + suffix);
    });
    def_find_defs(AST_SymbolDeclaration, function() {
        if (!this.global()) return;
    });
    def_find_defs(AST_SymbolRef, function(compressor, suffix) {
        if (!this.global()) return;
        var defines = compressor.option("global_defs");
        var name = this.name + suffix;
        if (HOP(defines, name)) return to_node(defines[name], this);
    });
})(function(node, func) {
    node.DEFMETHOD("_find_defs", func);
});

function best_of_expression(ast1, ast2) {
    return ast1.size() > ast2.size() ? ast2 : ast1;
}

function best_of_statement(ast1, ast2) {
    return best_of_expression(
        make_node(AST_SimpleStatement, ast1, {
            body: ast1
        }),
        make_node(AST_SimpleStatement, ast2, {
            body: ast2
        })
    ).body;
}

function best_of(compressor, ast1, ast2) {
    return (first_in_statement(compressor) ? best_of_statement : best_of_expression)(ast1, ast2);
}

function convert_to_predicate(obj) {
    const out = new Map();
    for (var key of Object.keys(obj)) {
        out.set(key, makePredicate(obj[key]));
    }
    return out;
}

var object_fns = [
    "constructor",
    "toString",
    "valueOf",
];
var native_fns = convert_to_predicate({
    Array: [
        "indexOf",
        "join",
        "lastIndexOf",
        "slice",
    ].concat(object_fns),
    Boolean: object_fns,
    Function: object_fns,
    Number: [
        "toExponential",
        "toFixed",
        "toPrecision",
    ].concat(object_fns),
    Object: object_fns,
    RegExp: [
        "test",
    ].concat(object_fns),
    String: [
        "charAt",
        "charCodeAt",
        "concat",
        "indexOf",
        "italics",
        "lastIndexOf",
        "match",
        "replace",
        "search",
        "slice",
        "split",
        "substr",
        "substring",
        "toLowerCase",
        "toUpperCase",
        "trim",
    ].concat(object_fns),
});
var static_fns = convert_to_predicate({
    Array: [
        "isArray",
    ],
    Math: [
        "abs",
        "acos",
        "asin",
        "atan",
        "ceil",
        "cos",
        "exp",
        "floor",
        "log",
        "round",
        "sin",
        "sqrt",
        "tan",
        "atan2",
        "pow",
        "max",
        "min",
    ],
    Number: [
        "isFinite",
        "isNaN",
    ],
    Object: [
        "create",
        "getOwnPropertyDescriptor",
        "getOwnPropertyNames",
        "getPrototypeOf",
        "isExtensible",
        "isFrozen",
        "isSealed",
        "keys",
    ],
    String: [
        "fromCharCode",
    ],
});

// methods to evaluate a constant expression
(function(def_eval) {
    // If the node has been successfully reduced to a constant,
    // then its value is returned; otherwise the element itself
    // is returned.
    // They can be distinguished as constant value is never a
    // descendant of AST_Node.
    AST_Node.DEFMETHOD("evaluate", function(compressor) {
        if (!compressor.option("evaluate")) return this;
        var val = this._eval(compressor, 1);
        if (!val || val instanceof RegExp) return val;
        if (typeof val == "function" || typeof val == "object") return this;
        return val;
    });
    var unaryPrefix = makePredicate("! ~ - + void");
    AST_Node.DEFMETHOD("is_constant", function() {
        // Accomodate when compress option evaluate=false
        // as well as the common constant expressions !0 and -1
        if (this instanceof AST_Constant) {
            return !(this instanceof AST_RegExp);
        } else {
            return this instanceof AST_UnaryPrefix
                && this.expression instanceof AST_Constant
                && unaryPrefix.has(this.operator);
        }
    });
    def_eval(AST_Statement, function() {
        throw new Error(string_template("Cannot evaluate a statement [{file}:{line},{col}]", this.start));
    });
    def_eval(AST_Lambda, return_this);
    def_eval(AST_Class, return_this);
    def_eval(AST_Node, return_this);
    def_eval(AST_Constant, function() {
        return this.getValue();
    });
    def_eval(AST_BigInt, return_this);
    def_eval(AST_RegExp, function(compressor) {
        let evaluated = compressor.evaluated_regexps.get(this);
        if (evaluated === undefined) {
            try {
                evaluated = (0, eval)(this.print_to_string());
            } catch (e) {
                evaluated = null;
            }
            compressor.evaluated_regexps.set(this, evaluated);
        }
        return evaluated || this;
    });
    def_eval(AST_TemplateString, function() {
        if (this.segments.length !== 1) return this;
        return this.segments[0].value;
    });
    def_eval(AST_Function, function(compressor) {
        if (compressor.option("unsafe")) {
            var fn = function() {};
            fn.node = this;
            fn.toString = () => this.print_to_string();
            return fn;
        }
        return this;
    });
    def_eval(AST_Array, function(compressor, depth) {
        if (compressor.option("unsafe")) {
            var elements = [];
            for (var i = 0, len = this.elements.length; i < len; i++) {
                var element = this.elements[i];
                var value = element._eval(compressor, depth);
                if (element === value) return this;
                elements.push(value);
            }
            return elements;
        }
        return this;
    });
    def_eval(AST_Object, function(compressor, depth) {
        if (compressor.option("unsafe")) {
            var val = {};
            for (var i = 0, len = this.properties.length; i < len; i++) {
                var prop = this.properties[i];
                if (prop instanceof AST_Expansion) return this;
                var key = prop.key;
                if (key instanceof AST_Symbol) {
                    key = key.name;
                } else if (key instanceof AST_Node) {
                    key = key._eval(compressor, depth);
                    if (key === prop.key) return this;
                }
                if (typeof Object.prototype[key] === "function") {
                    return this;
                }
                if (prop.value instanceof AST_Function) continue;
                val[key] = prop.value._eval(compressor, depth);
                if (val[key] === prop.value) return this;
            }
            return val;
        }
        return this;
    });
    var non_converting_unary = makePredicate("! typeof void");
    def_eval(AST_UnaryPrefix, function(compressor, depth) {
        var e = this.expression;
        // Function would be evaluated to an array and so typeof would
        // incorrectly return 'object'. Hence making is a special case.
        if (compressor.option("typeofs")
            && this.operator == "typeof"
            && (e instanceof AST_Lambda
                || e instanceof AST_SymbolRef
                    && e.fixed_value() instanceof AST_Lambda)) {
            return typeof function() {};
        }
        if (!non_converting_unary.has(this.operator)) depth++;
        e = e._eval(compressor, depth);
        if (e === this.expression) return this;
        switch (this.operator) {
          case "!": return !e;
          case "typeof":
            // typeof <RegExp> returns "object" or "function" on different platforms
            // so cannot evaluate reliably
            if (e instanceof RegExp) return this;
            return typeof e;
          case "void": return void e;
          case "~": return ~e;
          case "-": return -e;
          case "+": return +e;
        }
        return this;
    });
    var non_converting_binary = makePredicate("&& || ?? === !==");
    const identity_comparison = makePredicate("== != === !==");
    const has_identity = value =>
        typeof value === "object"
        || typeof value === "function"
        || typeof value === "symbol";

    def_eval(AST_Binary, function(compressor, depth) {
        if (!non_converting_binary.has(this.operator)) depth++;

        var left = this.left._eval(compressor, depth);
        if (left === this.left) return this;
        var right = this.right._eval(compressor, depth);
        if (right === this.right) return this;
        var result;

        if (
            left != null
            && right != null
            && identity_comparison.has(this.operator)
            && has_identity(left)
            && has_identity(right)
            && typeof left === typeof right
        ) {
            // Do not compare by reference
            return this;
        }

        switch (this.operator) {
          case "&&"  : result = left &&  right; break;
          case "||"  : result = left ||  right; break;
          case "??"  : result = left != null ? left : right; break;
          case "|"   : result = left |   right; break;
          case "&"   : result = left &   right; break;
          case "^"   : result = left ^   right; break;
          case "+"   : result = left +   right; break;
          case "*"   : result = left *   right; break;
          case "**"  : result = Math.pow(left, right); break;
          case "/"   : result = left /   right; break;
          case "%"   : result = left %   right; break;
          case "-"   : result = left -   right; break;
          case "<<"  : result = left <<  right; break;
          case ">>"  : result = left >>  right; break;
          case ">>>" : result = left >>> right; break;
          case "=="  : result = left ==  right; break;
          case "===" : result = left === right; break;
          case "!="  : result = left !=  right; break;
          case "!==" : result = left !== right; break;
          case "<"   : result = left <   right; break;
          case "<="  : result = left <=  right; break;
          case ">"   : result = left >   right; break;
          case ">="  : result = left >=  right; break;
          default:
              return this;
        }
        if (isNaN(result) && compressor.find_parent(AST_With)) {
            // leave original expression as is
            return this;
        }
        return result;
    });
    def_eval(AST_Conditional, function(compressor, depth) {
        var condition = this.condition._eval(compressor, depth);
        if (condition === this.condition) return this;
        var node = condition ? this.consequent : this.alternative;
        var value = node._eval(compressor, depth);
        return value === node ? this : value;
    });

    // Set of AST_SymbolRef which are currently being evaluated.
    // Avoids infinite recursion of ._eval()
    const reentrant_ref_eval = new Set();
    def_eval(AST_SymbolRef, function(compressor, depth) {
        if (reentrant_ref_eval.has(this)) return this;

        var fixed = this.fixed_value();
        if (!fixed) return this;

        reentrant_ref_eval.add(this);
        const value = fixed._eval(compressor, depth);
        reentrant_ref_eval.delete(this);

        if (value === fixed) return this;

        if (value && typeof value == "object") {
            var escaped = this.definition().escaped;
            if (escaped && depth > escaped) return this;
        }
        return value;
    });
    var global_objs = { Array, Math, Number, Object, String };
    var static_values = convert_to_predicate({
        Math: [
            "E",
            "LN10",
            "LN2",
            "LOG2E",
            "LOG10E",
            "PI",
            "SQRT1_2",
            "SQRT2",
        ],
        Number: [
            "MAX_VALUE",
            "MIN_VALUE",
            "NaN",
            "NEGATIVE_INFINITY",
            "POSITIVE_INFINITY",
        ],
    });
    const regexp_flags = new Set([
        "dotAll",
        "global",
        "ignoreCase",
        "multiline",
        "sticky",
        "unicode",
    ]);
    def_eval(AST_PropAccess, function(compressor, depth) {
        if (this.optional) {
            const obj = this.expression._eval(compressor, depth);
            if (obj == null) return undefined;
        }
        if (compressor.option("unsafe")) {
            var key = this.property;
            if (key instanceof AST_Node) {
                key = key._eval(compressor, depth);
                if (key === this.property) return this;
            }
            var exp = this.expression;
            var val;
            if (is_undeclared_ref(exp)) {

                var aa;
                var first_arg = exp.name === "hasOwnProperty"
                    && key === "call"
                    && (aa = compressor.parent() && compressor.parent().args)
                    && (aa && aa[0]
                    && aa[0].evaluate(compressor));

                first_arg = first_arg instanceof AST_Dot ? first_arg.expression : first_arg;

                if (first_arg == null || first_arg.thedef && first_arg.thedef.undeclared) {
                    return this.clone();
                }
                var static_value = static_values.get(exp.name);
                if (!static_value || !static_value.has(key)) return this;
                val = global_objs[exp.name];
            } else {
                val = exp._eval(compressor, depth + 1);
                if (val instanceof RegExp) {
                    if (key == "source") {
                        return regexp_source_fix(val.source);
                    } else if (key == "flags" || regexp_flags.has(key)) {
                        return val[key];
                    }
                }
                if (!val || val === exp || !HOP(val, key)) return this;
                if (typeof val == "function") switch (key) {
                  case "name":
                    return val.node.name ? val.node.name.name : "";
                  case "length":
                    return val.node.length_property();
                  default:
                    return this;
                }
            }
            return val[key];
        }
        return this;
    });
    def_eval(AST_Chain, function(compressor, depth) {
        const evaluated = this.expression._eval(compressor, depth);
        return evaluated === this.expression ? this : evaluated;
    });
    def_eval(AST_Call, function(compressor, depth) {
        var exp = this.expression;
        if (this.optional) {
            const callee = this.expression._eval(compressor, depth);
            if (callee == null) return undefined;
        }
        if (compressor.option("unsafe") && exp instanceof AST_PropAccess) {
            var key = exp.property;
            if (key instanceof AST_Node) {
                key = key._eval(compressor, depth);
                if (key === exp.property) return this;
            }
            var val;
            var e = exp.expression;
            if (is_undeclared_ref(e)) {
                var first_arg =
                    e.name === "hasOwnProperty" &&
                    key === "call" &&
                    (this.args[0] && this.args[0].evaluate(compressor));

                first_arg = first_arg instanceof AST_Dot ? first_arg.expression : first_arg;

                if ((first_arg == null || first_arg.thedef && first_arg.thedef.undeclared)) {
                    return this.clone();
                }
                var static_fn = static_fns.get(e.name);
                if (!static_fn || !static_fn.has(key)) return this;
                val = global_objs[e.name];
            } else {
                val = e._eval(compressor, depth + 1);
                if (val === e || !val) return this;
                var native_fn = native_fns.get(val.constructor.name);
                if (!native_fn || !native_fn.has(key)) return this;
            }
            var args = [];
            for (var i = 0, len = this.args.length; i < len; i++) {
                var arg = this.args[i];
                var value = arg._eval(compressor, depth);
                if (arg === value) return this;
                if (arg instanceof AST_Lambda) return this;
                args.push(value);
            }
            try {
                return val[key].apply(val, args);
            } catch (ex) {
                // We don't really care
            }
        }
        return this;
    });
    def_eval(AST_New, return_this);
})(function(node, func) {
    node.DEFMETHOD("_eval", func);
});

// method to negate an expression
(function(def_negate) {
    function basic_negation(exp) {
        return make_node(AST_UnaryPrefix, exp, {
            operator: "!",
            expression: exp
        });
    }
    function best(orig, alt, first_in_statement) {
        var negated = basic_negation(orig);
        if (first_in_statement) {
            var stat = make_node(AST_SimpleStatement, alt, {
                body: alt
            });
            return best_of_expression(negated, stat) === stat ? alt : negated;
        }
        return best_of_expression(negated, alt);
    }
    def_negate(AST_Node, function() {
        return basic_negation(this);
    });
    def_negate(AST_Statement, function() {
        throw new Error("Cannot negate a statement");
    });
    def_negate(AST_Function, function() {
        return basic_negation(this);
    });
    def_negate(AST_Arrow, function() {
        return basic_negation(this);
    });
    def_negate(AST_UnaryPrefix, function() {
        if (this.operator == "!")
            return this.expression;
        return basic_negation(this);
    });
    def_negate(AST_Sequence, function(compressor) {
        var expressions = this.expressions.slice();
        expressions.push(expressions.pop().negate(compressor));
        return make_sequence(this, expressions);
    });
    def_negate(AST_Conditional, function(compressor, first_in_statement) {
        var self = this.clone();
        self.consequent = self.consequent.negate(compressor);
        self.alternative = self.alternative.negate(compressor);
        return best(this, self, first_in_statement);
    });
    def_negate(AST_Binary, function(compressor, first_in_statement) {
        var self = this.clone(), op = this.operator;
        if (compressor.option("unsafe_comps")) {
            switch (op) {
              case "<=" : self.operator = ">"  ; return self;
              case "<"  : self.operator = ">=" ; return self;
              case ">=" : self.operator = "<"  ; return self;
              case ">"  : self.operator = "<=" ; return self;
            }
        }
        switch (op) {
          case "==" : self.operator = "!="; return self;
          case "!=" : self.operator = "=="; return self;
          case "===": self.operator = "!=="; return self;
          case "!==": self.operator = "==="; return self;
          case "&&":
            self.operator = "||";
            self.left = self.left.negate(compressor, first_in_statement);
            self.right = self.right.negate(compressor);
            return best(this, self, first_in_statement);
          case "||":
            self.operator = "&&";
            self.left = self.left.negate(compressor, first_in_statement);
            self.right = self.right.negate(compressor);
            return best(this, self, first_in_statement);
          case "??":
            self.right = self.right.negate(compressor);
            return best(this, self, first_in_statement);
        }
        return basic_negation(this);
    });
})(function(node, func) {
    node.DEFMETHOD("negate", function(compressor, first_in_statement) {
        return func.call(this, compressor, first_in_statement);
    });
});

var global_pure_fns = makePredicate("Boolean decodeURI decodeURIComponent Date encodeURI encodeURIComponent Error escape EvalError isFinite isNaN Number Object parseFloat parseInt RangeError ReferenceError String SyntaxError TypeError unescape URIError");
AST_Call.DEFMETHOD("is_callee_pure", function(compressor) {
    if (compressor.option("unsafe")) {
        var expr = this.expression;
        var first_arg = (this.args && this.args[0] && this.args[0].evaluate(compressor));
        if (
            expr.expression && expr.expression.name === "hasOwnProperty" &&
            (first_arg == null || first_arg.thedef && first_arg.thedef.undeclared)
        ) {
            return false;
        }
        if (is_undeclared_ref(expr) && global_pure_fns.has(expr.name)) return true;
        let static_fn;
        if (expr instanceof AST_Dot
            && is_undeclared_ref(expr.expression)
            && (static_fn = static_fns.get(expr.expression.name))
            && static_fn.has(expr.property)) {
            return true;
        }
    }
    return !!has_annotation(this, _PURE) || !compressor.pure_funcs(this);
});
AST_Node.DEFMETHOD("is_call_pure", return_false);
AST_Dot.DEFMETHOD("is_call_pure", function(compressor) {
    if (!compressor.option("unsafe")) return;
    const expr = this.expression;
    let map;
    if (expr instanceof AST_Array) {
        map = native_fns.get("Array");
    } else if (expr.is_boolean()) {
        map = native_fns.get("Boolean");
    } else if (expr.is_number(compressor)) {
        map = native_fns.get("Number");
    } else if (expr instanceof AST_RegExp) {
        map = native_fns.get("RegExp");
    } else if (expr.is_string(compressor)) {
        map = native_fns.get("String");
    } else if (!this.may_throw_on_access(compressor)) {
        map = native_fns.get("Object");
    }
    return map && map.has(this.property);
});

const pure_prop_access_globals = new Set([
    "Number",
    "String",
    "Array",
    "Object",
    "Function",
    "Promise",
]);
// determine if expression has side effects
(function(def_has_side_effects) {
    def_has_side_effects(AST_Node, return_true);

    def_has_side_effects(AST_EmptyStatement, return_false);
    def_has_side_effects(AST_Constant, return_false);
    def_has_side_effects(AST_This, return_false);

    function any(list, compressor) {
        for (var i = list.length; --i >= 0;)
            if (list[i].has_side_effects(compressor))
                return true;
        return false;
    }

    def_has_side_effects(AST_Block, function(compressor) {
        return any(this.body, compressor);
    });
    def_has_side_effects(AST_Call, function(compressor) {
        if (
            !this.is_callee_pure(compressor)
            && (!this.expression.is_call_pure(compressor)
                || this.expression.has_side_effects(compressor))
        ) {
            return true;
        }
        return any(this.args, compressor);
    });
    def_has_side_effects(AST_Switch, function(compressor) {
        return this.expression.has_side_effects(compressor)
            || any(this.body, compressor);
    });
    def_has_side_effects(AST_Case, function(compressor) {
        return this.expression.has_side_effects(compressor)
            || any(this.body, compressor);
    });
    def_has_side_effects(AST_Try, function(compressor) {
        return any(this.body, compressor)
            || this.bcatch && this.bcatch.has_side_effects(compressor)
            || this.bfinally && this.bfinally.has_side_effects(compressor);
    });
    def_has_side_effects(AST_If, function(compressor) {
        return this.condition.has_side_effects(compressor)
            || this.body && this.body.has_side_effects(compressor)
            || this.alternative && this.alternative.has_side_effects(compressor);
    });
    def_has_side_effects(AST_LabeledStatement, function(compressor) {
        return this.body.has_side_effects(compressor);
    });
    def_has_side_effects(AST_SimpleStatement, function(compressor) {
        return this.body.has_side_effects(compressor);
    });
    def_has_side_effects(AST_Lambda, return_false);
    def_has_side_effects(AST_Class, function (compressor) {
        if (this.extends && this.extends.has_side_effects(compressor)) {
            return true;
        }
        return any(this.properties, compressor);
    });
    def_has_side_effects(AST_Binary, function(compressor) {
        return this.left.has_side_effects(compressor)
            || this.right.has_side_effects(compressor);
    });
    def_has_side_effects(AST_Assign, return_true);
    def_has_side_effects(AST_Conditional, function(compressor) {
        return this.condition.has_side_effects(compressor)
            || this.consequent.has_side_effects(compressor)
            || this.alternative.has_side_effects(compressor);
    });
    def_has_side_effects(AST_Unary, function(compressor) {
        return unary_side_effects.has(this.operator)
            || this.expression.has_side_effects(compressor);
    });
    def_has_side_effects(AST_SymbolRef, function(compressor) {
        return !this.is_declared(compressor) && !pure_prop_access_globals.has(this.name);
    });
    def_has_side_effects(AST_SymbolClassProperty, return_false);
    def_has_side_effects(AST_SymbolDeclaration, return_false);
    def_has_side_effects(AST_Object, function(compressor) {
        return any(this.properties, compressor);
    });
    def_has_side_effects(AST_ObjectProperty, function(compressor) {
        return (
            this.computed_key() && this.key.has_side_effects(compressor)
            || this.value && this.value.has_side_effects(compressor)
        );
    });
    def_has_side_effects(AST_ClassProperty, function(compressor) {
        return (
            this.computed_key() && this.key.has_side_effects(compressor)
            || this.static && this.value && this.value.has_side_effects(compressor)
        );
    });
    def_has_side_effects(AST_ConciseMethod, function(compressor) {
        return this.computed_key() && this.key.has_side_effects(compressor);
    });
    def_has_side_effects(AST_ObjectGetter, function(compressor) {
        return this.computed_key() && this.key.has_side_effects(compressor);
    });
    def_has_side_effects(AST_ObjectSetter, function(compressor) {
        return this.computed_key() && this.key.has_side_effects(compressor);
    });
    def_has_side_effects(AST_Array, function(compressor) {
        return any(this.elements, compressor);
    });
    def_has_side_effects(AST_Dot, function(compressor) {
        return !this.optional && this.expression.may_throw_on_access(compressor)
            || this.expression.has_side_effects(compressor);
    });
    def_has_side_effects(AST_Sub, function(compressor) {
        if (this.optional && is_nullish(this.expression)) {
            return false;
        }

        return !this.optional && this.expression.may_throw_on_access(compressor)
            || this.expression.has_side_effects(compressor)
            || this.property.has_side_effects(compressor);
    });
    def_has_side_effects(AST_Chain, function (compressor) {
        return this.expression.has_side_effects(compressor);
    });
    def_has_side_effects(AST_Sequence, function(compressor) {
        return any(this.expressions, compressor);
    });
    def_has_side_effects(AST_Definitions, function(compressor) {
        return any(this.definitions, compressor);
    });
    def_has_side_effects(AST_VarDef, function() {
        return this.value;
    });
    def_has_side_effects(AST_TemplateSegment, return_false);
    def_has_side_effects(AST_TemplateString, function(compressor) {
        return any(this.segments, compressor);
    });
})(function(node, func) {
    node.DEFMETHOD("has_side_effects", func);
});

// determine if expression may throw
(function(def_may_throw) {
    def_may_throw(AST_Node, return_true);

    def_may_throw(AST_Constant, return_false);
    def_may_throw(AST_EmptyStatement, return_false);
    def_may_throw(AST_Lambda, return_false);
    def_may_throw(AST_SymbolDeclaration, return_false);
    def_may_throw(AST_This, return_false);

    function any(list, compressor) {
        for (var i = list.length; --i >= 0;)
            if (list[i].may_throw(compressor))
                return true;
        return false;
    }

    def_may_throw(AST_Class, function(compressor) {
        if (this.extends && this.extends.may_throw(compressor)) return true;
        return any(this.properties, compressor);
    });

    def_may_throw(AST_Array, function(compressor) {
        return any(this.elements, compressor);
    });
    def_may_throw(AST_Assign, function(compressor) {
        if (this.right.may_throw(compressor)) return true;
        if (!compressor.has_directive("use strict")
            && this.operator == "="
            && this.left instanceof AST_SymbolRef) {
            return false;
        }
        return this.left.may_throw(compressor);
    });
    def_may_throw(AST_Binary, function(compressor) {
        return this.left.may_throw(compressor)
            || this.right.may_throw(compressor);
    });
    def_may_throw(AST_Block, function(compressor) {
        return any(this.body, compressor);
    });
    def_may_throw(AST_Call, function(compressor) {
        if (this.optional && is_nullish(this.expression)) return false;
        if (any(this.args, compressor)) return true;
        if (this.is_callee_pure(compressor)) return false;
        if (this.expression.may_throw(compressor)) return true;
        return !(this.expression instanceof AST_Lambda)
            || any(this.expression.body, compressor);
    });
    def_may_throw(AST_Case, function(compressor) {
        return this.expression.may_throw(compressor)
            || any(this.body, compressor);
    });
    def_may_throw(AST_Conditional, function(compressor) {
        return this.condition.may_throw(compressor)
            || this.consequent.may_throw(compressor)
            || this.alternative.may_throw(compressor);
    });
    def_may_throw(AST_Definitions, function(compressor) {
        return any(this.definitions, compressor);
    });
    def_may_throw(AST_If, function(compressor) {
        return this.condition.may_throw(compressor)
            || this.body && this.body.may_throw(compressor)
            || this.alternative && this.alternative.may_throw(compressor);
    });
    def_may_throw(AST_LabeledStatement, function(compressor) {
        return this.body.may_throw(compressor);
    });
    def_may_throw(AST_Object, function(compressor) {
        return any(this.properties, compressor);
    });
    def_may_throw(AST_ObjectProperty, function(compressor) {
        // TODO key may throw too
        return this.value ? this.value.may_throw(compressor) : false;
    });
    def_may_throw(AST_ClassProperty, function(compressor) {
        return (
            this.computed_key() && this.key.may_throw(compressor)
            || this.static && this.value && this.value.may_throw(compressor)
        );
    });
    def_may_throw(AST_ConciseMethod, function(compressor) {
        return this.computed_key() && this.key.may_throw(compressor);
    });
    def_may_throw(AST_ObjectGetter, function(compressor) {
        return this.computed_key() && this.key.may_throw(compressor);
    });
    def_may_throw(AST_ObjectSetter, function(compressor) {
        return this.computed_key() && this.key.may_throw(compressor);
    });
    def_may_throw(AST_Return, function(compressor) {
        return this.value && this.value.may_throw(compressor);
    });
    def_may_throw(AST_Sequence, function(compressor) {
        return any(this.expressions, compressor);
    });
    def_may_throw(AST_SimpleStatement, function(compressor) {
        return this.body.may_throw(compressor);
    });
    def_may_throw(AST_Dot, function(compressor) {
        return !this.optional && this.expression.may_throw_on_access(compressor)
            || this.expression.may_throw(compressor);
    });
    def_may_throw(AST_Sub, function(compressor) {
        if (this.optional && is_nullish(this.expression)) return false;

        return !this.optional && this.expression.may_throw_on_access(compressor)
            || this.expression.may_throw(compressor)
            || this.property.may_throw(compressor);
    });
    def_may_throw(AST_Chain, function(compressor) {
        return this.expression.may_throw(compressor);
    });
    def_may_throw(AST_Switch, function(compressor) {
        return this.expression.may_throw(compressor)
            || any(this.body, compressor);
    });
    def_may_throw(AST_SymbolRef, function(compressor) {
        return !this.is_declared(compressor) && !pure_prop_access_globals.has(this.name);
    });
    def_may_throw(AST_SymbolClassProperty, return_false);
    def_may_throw(AST_Try, function(compressor) {
        return this.bcatch ? this.bcatch.may_throw(compressor) : any(this.body, compressor)
            || this.bfinally && this.bfinally.may_throw(compressor);
    });
    def_may_throw(AST_Unary, function(compressor) {
        if (this.operator == "typeof" && this.expression instanceof AST_SymbolRef)
            return false;
        return this.expression.may_throw(compressor);
    });
    def_may_throw(AST_VarDef, function(compressor) {
        if (!this.value) return false;
        return this.value.may_throw(compressor);
    });
})(function(node, func) {
    node.DEFMETHOD("may_throw", func);
});

// determine if expression is constant
(function(def_is_constant_expression) {
    function all_refs_local(scope) {
        let result = true;
        walk(this, node => {
            if (node instanceof AST_SymbolRef) {
                if (has_flag(this, INLINED)) {
                    result = false;
                    return walk_abort;
                }
                var def = node.definition();
                if (
                    member(def, this.enclosed)
                    && !this.variables.has(def.name)
                ) {
                    if (scope) {
                        var scope_def = scope.find_variable(node);
                        if (def.undeclared ? !scope_def : scope_def === def) {
                            result = "f";
                            return true;
                        }
                    }
                    result = false;
                    return walk_abort;
                }
                return true;
            }
            if (node instanceof AST_This && this instanceof AST_Arrow) {
                // TODO check arguments too!
                result = false;
                return walk_abort;
            }
        });
        return result;
    }

    def_is_constant_expression(AST_Node, return_false);
    def_is_constant_expression(AST_Constant, return_true);
    def_is_constant_expression(AST_Class, function(scope) {
        if (this.extends && !this.extends.is_constant_expression(scope)) {
            return false;
        }

        for (const prop of this.properties) {
            if (prop.computed_key() && !prop.key.is_constant_expression(scope)) {
                return false;
            }
            if (prop.static && prop.value && !prop.value.is_constant_expression(scope)) {
                return false;
            }
        }

        return all_refs_local.call(this, scope);
    });
    def_is_constant_expression(AST_Lambda, all_refs_local);
    def_is_constant_expression(AST_Unary, function() {
        return this.expression.is_constant_expression();
    });
    def_is_constant_expression(AST_Binary, function() {
        return this.left.is_constant_expression()
            && this.right.is_constant_expression();
    });
    def_is_constant_expression(AST_Array, function() {
        return this.elements.every((l) => l.is_constant_expression());
    });
    def_is_constant_expression(AST_Object, function() {
        return this.properties.every((l) => l.is_constant_expression());
    });
    def_is_constant_expression(AST_ObjectProperty, function() {
        return !!(!(this.key instanceof AST_Node) && this.value && this.value.is_constant_expression());
    });
})(function(node, func) {
    node.DEFMETHOD("is_constant_expression", func);
});

// tell me if a statement aborts
function aborts(thing) {
    return thing && thing.aborts();
}
(function(def_aborts) {
    def_aborts(AST_Statement, return_null);
    def_aborts(AST_Jump, return_this);
    function block_aborts() {
        for (var i = 0; i < this.body.length; i++) {
            if (aborts(this.body[i])) {
                return this.body[i];
            }
        }
        return null;
    }
    def_aborts(AST_Import, function() { return null; });
    def_aborts(AST_BlockStatement, block_aborts);
    def_aborts(AST_SwitchBranch, block_aborts);
    def_aborts(AST_If, function() {
        return this.alternative && aborts(this.body) && aborts(this.alternative) && this;
    });
})(function(node, func) {
    node.DEFMETHOD("aborts", func);
});

/* -----[ optimizers ]----- */

var directives = new Set(["use asm", "use strict"]);
def_optimize(AST_Directive, function(self, compressor) {
    if (compressor.option("directives")
        && (!directives.has(self.value) || compressor.has_directive(self.value) !== self)) {
        return make_node(AST_EmptyStatement, self);
    }
    return self;
});

def_optimize(AST_Debugger, function(self, compressor) {
    if (compressor.option("drop_debugger"))
        return make_node(AST_EmptyStatement, self);
    return self;
});

def_optimize(AST_LabeledStatement, function(self, compressor) {
    if (self.body instanceof AST_Break
        && compressor.loopcontrol_target(self.body) === self.body) {
        return make_node(AST_EmptyStatement, self);
    }
    return self.label.references.length == 0 ? self.body : self;
});

def_optimize(AST_Block, function(self, compressor) {
    tighten_body(self.body, compressor);
    return self;
});

function can_be_extracted_from_if_block(node) {
    return !(
        node instanceof AST_Const
        || node instanceof AST_Let
        || node instanceof AST_Class
    );
}

def_optimize(AST_BlockStatement, function(self, compressor) {
    tighten_body(self.body, compressor);
    switch (self.body.length) {
      case 1:
        if (!compressor.has_directive("use strict")
            && compressor.parent() instanceof AST_If
            && can_be_extracted_from_if_block(self.body[0])
            || can_be_evicted_from_block(self.body[0])) {
            return self.body[0];
        }
        break;
      case 0: return make_node(AST_EmptyStatement, self);
    }
    return self;
});

function opt_AST_Lambda(self, compressor) {
    tighten_body(self.body, compressor);
    if (compressor.option("side_effects")
        && self.body.length == 1
        && self.body[0] === compressor.has_directive("use strict")) {
        self.body.length = 0;
    }
    return self;
}
def_optimize(AST_Lambda, opt_AST_Lambda);

const r_keep_assign = /keep_assign/;
AST_Scope.DEFMETHOD("drop_unused", function(compressor) {
    if (!compressor.option("unused")) return;
    if (compressor.has_directive("use asm")) return;
    var self = this;
    if (self.pinned()) return;
    var drop_funcs = !(self instanceof AST_Toplevel) || compressor.toplevel.funcs;
    var drop_vars = !(self instanceof AST_Toplevel) || compressor.toplevel.vars;
    const assign_as_unused = r_keep_assign.test(compressor.option("unused")) ? return_false : function(node) {
        if (node instanceof AST_Assign
            && !node.logical
            && (has_flag(node, WRITE_ONLY) || node.operator == "=")
        ) {
            return node.left;
        }
        if (node instanceof AST_Unary && has_flag(node, WRITE_ONLY)) {
            return node.expression;
        }
    };
    var in_use_ids = new Map();
    var fixed_ids = new Map();
    if (self instanceof AST_Toplevel && compressor.top_retain) {
        self.variables.forEach(function(def) {
            if (compressor.top_retain(def) && !in_use_ids.has(def.id)) {
                in_use_ids.set(def.id, def);
            }
        });
    }
    var var_defs_by_id = new Map();
    var initializations = new Map();
    // pass 1: find out which symbols are directly used in
    // this scope (not in nested scopes).
    var scope = this;
    var tw = new TreeWalker(function(node, descend) {
        if (node instanceof AST_Lambda && node.uses_arguments && !tw.has_directive("use strict")) {
            node.argnames.forEach(function(argname) {
                if (!(argname instanceof AST_SymbolDeclaration)) return;
                var def = argname.definition();
                if (!in_use_ids.has(def.id)) {
                    in_use_ids.set(def.id, def);
                }
            });
        }
        if (node === self) return;
        if (node instanceof AST_Defun || node instanceof AST_DefClass) {
            var node_def = node.name.definition();
            const in_export = tw.parent() instanceof AST_Export;
            if (in_export || !drop_funcs && scope === self) {
                if (node_def.global && !in_use_ids.has(node_def.id)) {
                    in_use_ids.set(node_def.id, node_def);
                }
            }
            if (node instanceof AST_DefClass) {
                if (
                    node.extends
                    && (node.extends.has_side_effects(compressor)
                    || node.extends.may_throw(compressor))
                ) {
                    node.extends.walk(tw);
                }
                for (const prop of node.properties) {
                    if (
                        prop.has_side_effects(compressor) ||
                        prop.may_throw(compressor)
                    ) {
                        prop.walk(tw);
                    }
                }
            }
            map_add(initializations, node_def.id, node);
            return true; // don't go in nested scopes
        }
        if (node instanceof AST_SymbolFunarg && scope === self) {
            map_add(var_defs_by_id, node.definition().id, node);
        }
        if (node instanceof AST_Definitions && scope === self) {
            const in_export = tw.parent() instanceof AST_Export;
            node.definitions.forEach(function(def) {
                if (def.name instanceof AST_SymbolVar) {
                    map_add(var_defs_by_id, def.name.definition().id, def);
                }
                if (in_export || !drop_vars) {
                    walk(def.name, node => {
                        if (node instanceof AST_SymbolDeclaration) {
                            const def = node.definition();
                            if (
                                (in_export || def.global)
                                && !in_use_ids.has(def.id)
                            ) {
                                in_use_ids.set(def.id, def);
                            }
                        }
                    });
                }
                if (def.value) {
                    if (def.name instanceof AST_Destructuring) {
                        def.walk(tw);
                    } else {
                        var node_def = def.name.definition();
                        map_add(initializations, node_def.id, def.value);
                        if (!node_def.chained && def.name.fixed_value() === def.value) {
                            fixed_ids.set(node_def.id, def);
                        }
                    }
                    if (def.value.has_side_effects(compressor)) {
                        def.value.walk(tw);
                    }
                }
            });
            return true;
        }
        return scan_ref_scoped(node, descend);
    });
    self.walk(tw);
    // pass 2: for every used symbol we need to walk its
    // initialization code to figure out if it uses other
    // symbols (that may not be in_use).
    tw = new TreeWalker(scan_ref_scoped);
    in_use_ids.forEach(function (def) {
        var init = initializations.get(def.id);
        if (init) init.forEach(function(init) {
            init.walk(tw);
        });
    });
    // pass 3: we should drop declarations not in_use
    var tt = new TreeTransformer(
        function before(node, descend, in_list) {
            var parent = tt.parent();
            if (drop_vars) {
                const sym = assign_as_unused(node);
                if (sym instanceof AST_SymbolRef) {
                    var def = sym.definition();
                    var in_use = in_use_ids.has(def.id);
                    if (node instanceof AST_Assign) {
                        if (!in_use || fixed_ids.has(def.id) && fixed_ids.get(def.id) !== node) {
                            return maintain_this_binding(parent, node, node.right.transform(tt));
                        }
                    } else if (!in_use) return in_list ? MAP.skip : make_node(AST_Number, node, {
                        value: 0
                    });
                }
            }
            if (scope !== self) return;
            var def;
            if (node.name
                && (node instanceof AST_ClassExpression
                    && !keep_name(compressor.option("keep_classnames"), (def = node.name.definition()).name)
                || node instanceof AST_Function
                    && !keep_name(compressor.option("keep_fnames"), (def = node.name.definition()).name))) {
                // any declarations with same name will overshadow
                // name of this anonymous function and can therefore
                // never be used anywhere
                if (!in_use_ids.has(def.id) || def.orig.length > 1) node.name = null;
            }
            if (node instanceof AST_Lambda && !(node instanceof AST_Accessor)) {
                var trim = !compressor.option("keep_fargs");
                for (var a = node.argnames, i = a.length; --i >= 0;) {
                    var sym = a[i];
                    if (sym instanceof AST_Expansion) {
                        sym = sym.expression;
                    }
                    if (sym instanceof AST_DefaultAssign) {
                        sym = sym.left;
                    }
                    // Do not drop destructuring arguments.
                    // They constitute a type assertion, so dropping
                    // them would stop that TypeError which would happen
                    // if someone called it with an incorrectly formatted
                    // parameter.
                    if (!(sym instanceof AST_Destructuring) && !in_use_ids.has(sym.definition().id)) {
                        set_flag(sym, UNUSED);
                        if (trim) {
                            a.pop();
                        }
                    } else {
                        trim = false;
                    }
                }
            }
            if ((node instanceof AST_Defun || node instanceof AST_DefClass) && node !== self) {
                const def = node.name.definition();
                let keep = def.global && !drop_funcs || in_use_ids.has(def.id);
                if (!keep) {
                    def.eliminated++;
                    if (node instanceof AST_DefClass) {
                        // Classes might have extends with side effects
                        const side_effects = node.drop_side_effect_free(compressor);
                        if (side_effects) {
                            return make_node(AST_SimpleStatement, node, {
                                body: side_effects
                            });
                        }
                    }
                    return in_list ? MAP.skip : make_node(AST_EmptyStatement, node);
                }
            }
            if (node instanceof AST_Definitions && !(parent instanceof AST_ForIn && parent.init === node)) {
                var drop_block = !(parent instanceof AST_Toplevel) && !(node instanceof AST_Var);
                // place uninitialized names at the start
                var body = [], head = [], tail = [];
                // for unused names whose initialization has
                // side effects, we can cascade the init. code
                // into the next one, or next statement.
                var side_effects = [];
                node.definitions.forEach(function(def) {
                    if (def.value) def.value = def.value.transform(tt);
                    var is_destructure = def.name instanceof AST_Destructuring;
                    var sym = is_destructure
                        ? new SymbolDef(null, { name: "<destructure>" }) /* fake SymbolDef */
                        : def.name.definition();
                    if (drop_block && sym.global) return tail.push(def);
                    if (!(drop_vars || drop_block)
                        || is_destructure
                            && (def.name.names.length
                                || def.name.is_array
                                || compressor.option("pure_getters") != true)
                        || in_use_ids.has(sym.id)
                    ) {
                        if (def.value && fixed_ids.has(sym.id) && fixed_ids.get(sym.id) !== def) {
                            def.value = def.value.drop_side_effect_free(compressor);
                        }
                        if (def.name instanceof AST_SymbolVar) {
                            var var_defs = var_defs_by_id.get(sym.id);
                            if (var_defs.length > 1 && (!def.value || sym.orig.indexOf(def.name) > sym.eliminated)) {
                                if (def.value) {
                                    var ref = make_node(AST_SymbolRef, def.name, def.name);
                                    sym.references.push(ref);
                                    var assign = make_node(AST_Assign, def, {
                                        operator: "=",
                                        logical: false,
                                        left: ref,
                                        right: def.value
                                    });
                                    if (fixed_ids.get(sym.id) === def) {
                                        fixed_ids.set(sym.id, assign);
                                    }
                                    side_effects.push(assign.transform(tt));
                                }
                                remove(var_defs, def);
                                sym.eliminated++;
                                return;
                            }
                        }
                        if (def.value) {
                            if (side_effects.length > 0) {
                                if (tail.length > 0) {
                                    side_effects.push(def.value);
                                    def.value = make_sequence(def.value, side_effects);
                                } else {
                                    body.push(make_node(AST_SimpleStatement, node, {
                                        body: make_sequence(node, side_effects)
                                    }));
                                }
                                side_effects = [];
                            }
                            tail.push(def);
                        } else {
                            head.push(def);
                        }
                    } else if (sym.orig[0] instanceof AST_SymbolCatch) {
                        var value = def.value && def.value.drop_side_effect_free(compressor);
                        if (value) side_effects.push(value);
                        def.value = null;
                        head.push(def);
                    } else {
                        var value = def.value && def.value.drop_side_effect_free(compressor);
                        if (value) {
                            side_effects.push(value);
                        }
                        sym.eliminated++;
                    }
                });
                if (head.length > 0 || tail.length > 0) {
                    node.definitions = head.concat(tail);
                    body.push(node);
                }
                if (side_effects.length > 0) {
                    body.push(make_node(AST_SimpleStatement, node, {
                        body: make_sequence(node, side_effects)
                    }));
                }
                switch (body.length) {
                  case 0:
                    return in_list ? MAP.skip : make_node(AST_EmptyStatement, node);
                  case 1:
                    return body[0];
                  default:
                    return in_list ? MAP.splice(body) : make_node(AST_BlockStatement, node, {
                        body: body
                    });
                }
            }
            // certain combination of unused name + side effect leads to:
            //    https://github.com/mishoo/UglifyJS2/issues/44
            //    https://github.com/mishoo/UglifyJS2/issues/1830
            //    https://github.com/mishoo/UglifyJS2/issues/1838
            // that's an invalid AST.
            // We fix it at this stage by moving the `var` outside the `for`.
            if (node instanceof AST_For) {
                descend(node, this);
                var block;
                if (node.init instanceof AST_BlockStatement) {
                    block = node.init;
                    node.init = block.body.pop();
                    block.body.push(node);
                }
                if (node.init instanceof AST_SimpleStatement) {
                    node.init = node.init.body;
                } else if (is_empty(node.init)) {
                    node.init = null;
                }
                return !block ? node : in_list ? MAP.splice(block.body) : block;
            }
            if (node instanceof AST_LabeledStatement
                && node.body instanceof AST_For
            ) {
                descend(node, this);
                if (node.body instanceof AST_BlockStatement) {
                    var block = node.body;
                    node.body = block.body.pop();
                    block.body.push(node);
                    return in_list ? MAP.splice(block.body) : block;
                }
                return node;
            }
            if (node instanceof AST_BlockStatement) {
                descend(node, this);
                if (in_list && node.body.every(can_be_evicted_from_block)) {
                    return MAP.splice(node.body);
                }
                return node;
            }
            if (node instanceof AST_Scope) {
                const save_scope = scope;
                scope = node;
                descend(node, this);
                scope = save_scope;
                return node;
            }
        }
    );

    self.transform(tt);

    function scan_ref_scoped(node, descend) {
        var node_def;
        const sym = assign_as_unused(node);
        if (sym instanceof AST_SymbolRef
            && !is_ref_of(node.left, AST_SymbolBlockDeclaration)
            && self.variables.get(sym.name) === (node_def = sym.definition())
        ) {
            if (node instanceof AST_Assign) {
                node.right.walk(tw);
                if (!node_def.chained && node.left.fixed_value() === node.right) {
                    fixed_ids.set(node_def.id, node);
                }
            }
            return true;
        }
        if (node instanceof AST_SymbolRef) {
            node_def = node.definition();
            if (!in_use_ids.has(node_def.id)) {
                in_use_ids.set(node_def.id, node_def);
                if (node_def.orig[0] instanceof AST_SymbolCatch) {
                    const redef = node_def.scope.is_block_scope()
                        && node_def.scope.get_defun_scope().variables.get(node_def.name);
                    if (redef) in_use_ids.set(redef.id, redef);
                }
            }
            return true;
        }
        if (node instanceof AST_Scope) {
            var save_scope = scope;
            scope = node;
            descend();
            scope = save_scope;
            return true;
        }
    }
});

AST_Scope.DEFMETHOD("hoist_declarations", function(compressor) {
    var self = this;
    if (compressor.has_directive("use asm")) return self;
    // Hoisting makes no sense in an arrow func
    if (!Array.isArray(self.body)) return self;

    var hoist_funs = compressor.option("hoist_funs");
    var hoist_vars = compressor.option("hoist_vars");

    if (hoist_funs || hoist_vars) {
        var dirs = [];
        var hoisted = [];
        var vars = new Map(), vars_found = 0, var_decl = 0;
        // let's count var_decl first, we seem to waste a lot of
        // space if we hoist `var` when there's only one.
        walk(self, node => {
            if (node instanceof AST_Scope && node !== self)
                return true;
            if (node instanceof AST_Var) {
                ++var_decl;
                return true;
            }
        });
        hoist_vars = hoist_vars && var_decl > 1;
        var tt = new TreeTransformer(
            function before(node) {
                if (node !== self) {
                    if (node instanceof AST_Directive) {
                        dirs.push(node);
                        return make_node(AST_EmptyStatement, node);
                    }
                    if (hoist_funs && node instanceof AST_Defun
                        && !(tt.parent() instanceof AST_Export)
                        && tt.parent() === self) {
                        hoisted.push(node);
                        return make_node(AST_EmptyStatement, node);
                    }
                    if (
                        hoist_vars
                        && node instanceof AST_Var
                        && !node.definitions.some(def => def.name instanceof AST_Destructuring)
                    ) {
                        node.definitions.forEach(function(def) {
                            vars.set(def.name.name, def);
                            ++vars_found;
                        });
                        var seq = node.to_assignments(compressor);
                        var p = tt.parent();
                        if (p instanceof AST_ForIn && p.init === node) {
                            if (seq == null) {
                                var def = node.definitions[0].name;
                                return make_node(AST_SymbolRef, def, def);
                            }
                            return seq;
                        }
                        if (p instanceof AST_For && p.init === node) {
                            return seq;
                        }
                        if (!seq) return make_node(AST_EmptyStatement, node);
                        return make_node(AST_SimpleStatement, node, {
                            body: seq
                        });
                    }
                    if (node instanceof AST_Scope)
                        return node; // to avoid descending in nested scopes
                }
            }
        );
        self = self.transform(tt);
        if (vars_found > 0) {
            // collect only vars which don't show up in self's arguments list
            var defs = [];
            const is_lambda = self instanceof AST_Lambda;
            const args_as_names = is_lambda ? self.args_as_names() : null;
            vars.forEach((def, name) => {
                if (is_lambda && args_as_names.some((x) => x.name === def.name.name)) {
                    vars.delete(name);
                } else {
                    def = def.clone();
                    def.value = null;
                    defs.push(def);
                    vars.set(name, def);
                }
            });
            if (defs.length > 0) {
                // try to merge in assignments
                for (var i = 0; i < self.body.length;) {
                    if (self.body[i] instanceof AST_SimpleStatement) {
                        var expr = self.body[i].body, sym, assign;
                        if (expr instanceof AST_Assign
                            && expr.operator == "="
                            && (sym = expr.left) instanceof AST_Symbol
                            && vars.has(sym.name)
                        ) {
                            var def = vars.get(sym.name);
                            if (def.value) break;
                            def.value = expr.right;
                            remove(defs, def);
                            defs.push(def);
                            self.body.splice(i, 1);
                            continue;
                        }
                        if (expr instanceof AST_Sequence
                            && (assign = expr.expressions[0]) instanceof AST_Assign
                            && assign.operator == "="
                            && (sym = assign.left) instanceof AST_Symbol
                            && vars.has(sym.name)
                        ) {
                            var def = vars.get(sym.name);
                            if (def.value) break;
                            def.value = assign.right;
                            remove(defs, def);
                            defs.push(def);
                            self.body[i].body = make_sequence(expr, expr.expressions.slice(1));
                            continue;
                        }
                    }
                    if (self.body[i] instanceof AST_EmptyStatement) {
                        self.body.splice(i, 1);
                        continue;
                    }
                    if (self.body[i] instanceof AST_BlockStatement) {
                        self.body.splice(i, 1, ...self.body[i].body);
                        continue;
                    }
                    break;
                }
                defs = make_node(AST_Var, self, {
                    definitions: defs
                });
                hoisted.push(defs);
            }
        }
        self.body = dirs.concat(hoisted, self.body);
    }
    return self;
});

AST_Scope.DEFMETHOD("hoist_properties", function(compressor) {
    var self = this;
    if (!compressor.option("hoist_props") || compressor.has_directive("use asm")) return self;
    var top_retain = self instanceof AST_Toplevel && compressor.top_retain || return_false;
    var defs_by_id = new Map();
    var hoister = new TreeTransformer(function(node, descend) {
        if (node instanceof AST_Definitions
            && hoister.parent() instanceof AST_Export) return node;
        if (node instanceof AST_VarDef) {
            const sym = node.name;
            let def;
            let value;
            if (sym.scope === self
                && (def = sym.definition()).escaped != 1
                && !def.assignments
                && !def.direct_access
                && !def.single_use
                && !compressor.exposed(def)
                && !top_retain(def)
                && (value = sym.fixed_value()) === node.value
                && value instanceof AST_Object
                && !value.properties.some(prop =>
                    prop instanceof AST_Expansion || prop.computed_key()
                )
            ) {
                descend(node, this);
                const defs = new Map();
                const assignments = [];
                value.properties.forEach(({ key, value }) => {
                    const scope = find_scope(hoister);
                    const symbol = self.create_symbol(sym.CTOR, {
                        source: sym,
                        scope,
                        conflict_scopes: new Set([
                            scope,
                            ...sym.definition().references.map(ref => ref.scope)
                        ]),
                        tentative_name: sym.name + "_" + key
                    });

                    defs.set(String(key), symbol.definition());

                    assignments.push(make_node(AST_VarDef, node, {
                        name: symbol,
                        value
                    }));
                });
                defs_by_id.set(def.id, defs);
                return MAP.splice(assignments);
            }
        } else if (node instanceof AST_PropAccess
            && node.expression instanceof AST_SymbolRef
        ) {
            const defs = defs_by_id.get(node.expression.definition().id);
            if (defs) {
                const def = defs.get(String(get_value(node.property)));
                const sym = make_node(AST_SymbolRef, node, {
                    name: def.name,
                    scope: node.expression.scope,
                    thedef: def
                });
                sym.reference({});
                return sym;
            }
        }
    });
    return self.transform(hoister);
});

// drop_side_effect_free()
// remove side-effect-free parts which only affects return value
(function(def_drop_side_effect_free) {
    // Drop side-effect-free elements from an array of expressions.
    // Returns an array of expressions with side-effects or null
    // if all elements were dropped. Note: original array may be
    // returned if nothing changed.
    function trim(nodes, compressor, first_in_statement) {
        var len = nodes.length;
        if (!len) return null;
        var ret = [], changed = false;
        for (var i = 0; i < len; i++) {
            var node = nodes[i].drop_side_effect_free(compressor, first_in_statement);
            changed |= node !== nodes[i];
            if (node) {
                ret.push(node);
                first_in_statement = false;
            }
        }
        return changed ? ret.length ? ret : null : nodes;
    }

    def_drop_side_effect_free(AST_Node, return_this);
    def_drop_side_effect_free(AST_Constant, return_null);
    def_drop_side_effect_free(AST_This, return_null);
    def_drop_side_effect_free(AST_Call, function(compressor, first_in_statement) {
        if (this.optional && is_nullish(this.expression)) {
            return make_node(AST_Undefined, this);
        }

        if (!this.is_callee_pure(compressor)) {
            if (this.expression.is_call_pure(compressor)) {
                var exprs = this.args.slice();
                exprs.unshift(this.expression.expression);
                exprs = trim(exprs, compressor, first_in_statement);
                return exprs && make_sequence(this, exprs);
            }
            if (is_func_expr(this.expression)
                && (!this.expression.name || !this.expression.name.definition().references.length)) {
                var node = this.clone();
                node.expression.process_expression(false, compressor);
                return node;
            }
            return this;
        }
        var args = trim(this.args, compressor, first_in_statement);
        return args && make_sequence(this, args);
    });
    def_drop_side_effect_free(AST_Accessor, return_null);
    def_drop_side_effect_free(AST_Function, return_null);
    def_drop_side_effect_free(AST_Arrow, return_null);
    def_drop_side_effect_free(AST_Class, function (compressor) {
        const with_effects = [];
        const trimmed_extends = this.extends && this.extends.drop_side_effect_free(compressor);
        if (trimmed_extends) with_effects.push(trimmed_extends);
        for (const prop of this.properties) {
            const trimmed_prop = prop.drop_side_effect_free(compressor);
            if (trimmed_prop) with_effects.push(trimmed_prop);
        }
        if (!with_effects.length) return null;
        return make_sequence(this, with_effects);
    });
    def_drop_side_effect_free(AST_Binary, function(compressor, first_in_statement) {
        var right = this.right.drop_side_effect_free(compressor);
        if (!right) return this.left.drop_side_effect_free(compressor, first_in_statement);
        if (lazy_op.has(this.operator)) {
            if (right === this.right) return this;
            var node = this.clone();
            node.right = right;
            return node;
        } else {
            var left = this.left.drop_side_effect_free(compressor, first_in_statement);
            if (!left) return this.right.drop_side_effect_free(compressor, first_in_statement);
            return make_sequence(this, [ left, right ]);
        }
    });
    def_drop_side_effect_free(AST_Assign, function(compressor) {
        if (this.logical) return this;

        var left = this.left;
        if (left.has_side_effects(compressor)
            || compressor.has_directive("use strict")
                && left instanceof AST_PropAccess
                && left.expression.is_constant()) {
            return this;
        }
        set_flag(this, WRITE_ONLY);
        while (left instanceof AST_PropAccess) {
            left = left.expression;
        }
        if (left.is_constant_expression(compressor.find_parent(AST_Scope))) {
            return this.right.drop_side_effect_free(compressor);
        }
        return this;
    });
    def_drop_side_effect_free(AST_Conditional, function(compressor) {
        var consequent = this.consequent.drop_side_effect_free(compressor);
        var alternative = this.alternative.drop_side_effect_free(compressor);
        if (consequent === this.consequent && alternative === this.alternative) return this;
        if (!consequent) return alternative ? make_node(AST_Binary, this, {
            operator: "||",
            left: this.condition,
            right: alternative
        }) : this.condition.drop_side_effect_free(compressor);
        if (!alternative) return make_node(AST_Binary, this, {
            operator: "&&",
            left: this.condition,
            right: consequent
        });
        var node = this.clone();
        node.consequent = consequent;
        node.alternative = alternative;
        return node;
    });
    def_drop_side_effect_free(AST_Unary, function(compressor, first_in_statement) {
        if (unary_side_effects.has(this.operator)) {
            if (!this.expression.has_side_effects(compressor)) {
                set_flag(this, WRITE_ONLY);
            } else {
                clear_flag(this, WRITE_ONLY);
            }
            return this;
        }
        if (this.operator == "typeof" && this.expression instanceof AST_SymbolRef) return null;
        var expression = this.expression.drop_side_effect_free(compressor, first_in_statement);
        if (first_in_statement && expression && is_iife_call(expression)) {
            if (expression === this.expression && this.operator == "!") return this;
            return expression.negate(compressor, first_in_statement);
        }
        return expression;
    });
    def_drop_side_effect_free(AST_SymbolRef, function(compressor) {
        const safe_access = this.is_declared(compressor)
            || pure_prop_access_globals.has(this.name);
        return safe_access ? null : this;
    });
    def_drop_side_effect_free(AST_Object, function(compressor, first_in_statement) {
        var values = trim(this.properties, compressor, first_in_statement);
        return values && make_sequence(this, values);
    });
    def_drop_side_effect_free(AST_ObjectProperty, function(compressor, first_in_statement) {
        const computed_key = this instanceof AST_ObjectKeyVal && this.key instanceof AST_Node;
        const key = computed_key && this.key.drop_side_effect_free(compressor, first_in_statement);
        const value = this.value && this.value.drop_side_effect_free(compressor, first_in_statement);
        if (key && value) {
            return make_sequence(this, [key, value]);
        }
        return key || value;
    });
    def_drop_side_effect_free(AST_ClassProperty, function (compressor) {
        const key = this.computed_key() && this.key.drop_side_effect_free(compressor);

        const value = this.static && this.value
            && this.value.drop_side_effect_free(compressor);

        if (key && value) return make_sequence(this, [key, value]);
        return key || value || null;
    });
    def_drop_side_effect_free(AST_ConciseMethod, function () {
        return this.computed_key() ? this.key : null;
    });
    def_drop_side_effect_free(AST_ObjectGetter, function () {
        return this.computed_key() ? this.key : null;
    });
    def_drop_side_effect_free(AST_ObjectSetter, function () {
        return this.computed_key() ? this.key : null;
    });
    def_drop_side_effect_free(AST_Array, function(compressor, first_in_statement) {
        var values = trim(this.elements, compressor, first_in_statement);
        return values && make_sequence(this, values);
    });
    def_drop_side_effect_free(AST_Dot, function(compressor, first_in_statement) {
        if (this.optional) {
            return is_nullish(this.expression) ? make_node(AST_Undefined, this) : this;
        }
        if (this.expression.may_throw_on_access(compressor)) return this;

        return this.expression.drop_side_effect_free(compressor, first_in_statement);
    });
    def_drop_side_effect_free(AST_Sub, function(compressor, first_in_statement) {
        if (this.optional) {
            return is_nullish(this.expression) ? make_node(AST_Undefined, this): this;
        }
        if (this.expression.may_throw_on_access(compressor)) return this;

        var expression = this.expression.drop_side_effect_free(compressor, first_in_statement);
        if (!expression) return this.property.drop_side_effect_free(compressor, first_in_statement);
        var property = this.property.drop_side_effect_free(compressor);
        if (!property) return expression;
        return make_sequence(this, [ expression, property ]);
    });
    def_drop_side_effect_free(AST_Chain, function (compressor, first_in_statement) {
        return this.expression.drop_side_effect_free(compressor, first_in_statement);
    });
    def_drop_side_effect_free(AST_Sequence, function(compressor) {
        var last = this.tail_node();
        var expr = last.drop_side_effect_free(compressor);
        if (expr === last) return this;
        var expressions = this.expressions.slice(0, -1);
        if (expr) expressions.push(expr);
        if (!expressions.length) {
            return make_node(AST_Number, this, { value: 0 });
        }
        return make_sequence(this, expressions);
    });
    def_drop_side_effect_free(AST_Expansion, function(compressor, first_in_statement) {
        return this.expression.drop_side_effect_free(compressor, first_in_statement);
    });
    def_drop_side_effect_free(AST_TemplateSegment, return_null);
    def_drop_side_effect_free(AST_TemplateString, function(compressor) {
        var values = trim(this.segments, compressor, first_in_statement);
        return values && make_sequence(this, values);
    });
})(function(node, func) {
    node.DEFMETHOD("drop_side_effect_free", func);
});

def_optimize(AST_SimpleStatement, function(self, compressor) {
    if (compressor.option("side_effects")) {
        var body = self.body;
        var node = body.drop_side_effect_free(compressor, true);
        if (!node) {
            return make_node(AST_EmptyStatement, self);
        }
        if (node !== body) {
            return make_node(AST_SimpleStatement, self, { body: node });
        }
    }
    return self;
});

def_optimize(AST_While, function(self, compressor) {
    return compressor.option("loops") ? make_node(AST_For, self, self).optimize(compressor) : self;
});

function has_break_or_continue(loop, parent) {
    var found = false;
    var tw = new TreeWalker(function(node) {
        if (found || node instanceof AST_Scope) return true;
        if (node instanceof AST_LoopControl && tw.loopcontrol_target(node) === loop) {
            return found = true;
        }
    });
    if (parent instanceof AST_LabeledStatement) tw.push(parent);
    tw.push(loop);
    loop.body.walk(tw);
    return found;
}

def_optimize(AST_Do, function(self, compressor) {
    if (!compressor.option("loops")) return self;
    var cond = self.condition.tail_node().evaluate(compressor);
    if (!(cond instanceof AST_Node)) {
        if (cond) return make_node(AST_For, self, {
            body: make_node(AST_BlockStatement, self.body, {
                body: [
                    self.body,
                    make_node(AST_SimpleStatement, self.condition, {
                        body: self.condition
                    })
                ]
            })
        }).optimize(compressor);
        if (!has_break_or_continue(self, compressor.parent())) {
            return make_node(AST_BlockStatement, self.body, {
                body: [
                    self.body,
                    make_node(AST_SimpleStatement, self.condition, {
                        body: self.condition
                    })
                ]
            }).optimize(compressor);
        }
    }
    return self;
});

function if_break_in_loop(self, compressor) {
    var first = self.body instanceof AST_BlockStatement ? self.body.body[0] : self.body;
    if (compressor.option("dead_code") && is_break(first)) {
        var body = [];
        if (self.init instanceof AST_Statement) {
            body.push(self.init);
        } else if (self.init) {
            body.push(make_node(AST_SimpleStatement, self.init, {
                body: self.init
            }));
        }
        if (self.condition) {
            body.push(make_node(AST_SimpleStatement, self.condition, {
                body: self.condition
            }));
        }
        trim_unreachable_code(compressor, self.body, body);
        return make_node(AST_BlockStatement, self, {
            body: body
        });
    }
    if (first instanceof AST_If) {
        if (is_break(first.body)) {
            if (self.condition) {
                self.condition = make_node(AST_Binary, self.condition, {
                    left: self.condition,
                    operator: "&&",
                    right: first.condition.negate(compressor),
                });
            } else {
                self.condition = first.condition.negate(compressor);
            }
            drop_it(first.alternative);
        } else if (is_break(first.alternative)) {
            if (self.condition) {
                self.condition = make_node(AST_Binary, self.condition, {
                    left: self.condition,
                    operator: "&&",
                    right: first.condition,
                });
            } else {
                self.condition = first.condition;
            }
            drop_it(first.body);
        }
    }
    return self;

    function is_break(node) {
        return node instanceof AST_Break
            && compressor.loopcontrol_target(node) === compressor.self();
    }

    function drop_it(rest) {
        rest = as_statement_array(rest);
        if (self.body instanceof AST_BlockStatement) {
            self.body = self.body.clone();
            self.body.body = rest.concat(self.body.body.slice(1));
            self.body = self.body.transform(compressor);
        } else {
            self.body = make_node(AST_BlockStatement, self.body, {
                body: rest
            }).transform(compressor);
        }
        self = if_break_in_loop(self, compressor);
    }
}

def_optimize(AST_For, function(self, compressor) {
    if (!compressor.option("loops")) return self;
    if (compressor.option("side_effects") && self.init) {
        self.init = self.init.drop_side_effect_free(compressor);
    }
    if (self.condition) {
        var cond = self.condition.evaluate(compressor);
        if (!(cond instanceof AST_Node)) {
            if (cond) self.condition = null;
            else if (!compressor.option("dead_code")) {
                var orig = self.condition;
                self.condition = make_node_from_constant(cond, self.condition);
                self.condition = best_of_expression(self.condition.transform(compressor), orig);
            }
        }
        if (compressor.option("dead_code")) {
            if (cond instanceof AST_Node) cond = self.condition.tail_node().evaluate(compressor);
            if (!cond) {
                var body = [];
                trim_unreachable_code(compressor, self.body, body);
                if (self.init instanceof AST_Statement) {
                    body.push(self.init);
                } else if (self.init) {
                    body.push(make_node(AST_SimpleStatement, self.init, {
                        body: self.init
                    }));
                }
                body.push(make_node(AST_SimpleStatement, self.condition, {
                    body: self.condition
                }));
                return make_node(AST_BlockStatement, self, { body: body }).optimize(compressor);
            }
        }
    }
    return if_break_in_loop(self, compressor);
});

def_optimize(AST_If, function(self, compressor) {
    if (is_empty(self.alternative)) self.alternative = null;

    if (!compressor.option("conditionals")) return self;
    // if condition can be statically determined, drop
    // one of the blocks.  note, statically determined implies
    // “has no side effects”; also it doesn't work for cases like
    // `x && true`, though it probably should.
    var cond = self.condition.evaluate(compressor);
    if (!compressor.option("dead_code") && !(cond instanceof AST_Node)) {
        var orig = self.condition;
        self.condition = make_node_from_constant(cond, orig);
        self.condition = best_of_expression(self.condition.transform(compressor), orig);
    }
    if (compressor.option("dead_code")) {
        if (cond instanceof AST_Node) cond = self.condition.tail_node().evaluate(compressor);
        if (!cond) {
            var body = [];
            trim_unreachable_code(compressor, self.body, body);
            body.push(make_node(AST_SimpleStatement, self.condition, {
                body: self.condition
            }));
            if (self.alternative) body.push(self.alternative);
            return make_node(AST_BlockStatement, self, { body: body }).optimize(compressor);
        } else if (!(cond instanceof AST_Node)) {
            var body = [];
            body.push(make_node(AST_SimpleStatement, self.condition, {
                body: self.condition
            }));
            body.push(self.body);
            if (self.alternative) {
                trim_unreachable_code(compressor, self.alternative, body);
            }
            return make_node(AST_BlockStatement, self, { body: body }).optimize(compressor);
        }
    }
    var negated = self.condition.negate(compressor);
    var self_condition_length = self.condition.size();
    var negated_length = negated.size();
    var negated_is_best = negated_length < self_condition_length;
    if (self.alternative && negated_is_best) {
        negated_is_best = false; // because we already do the switch here.
        // no need to swap values of self_condition_length and negated_length
        // here because they are only used in an equality comparison later on.
        self.condition = negated;
        var tmp = self.body;
        self.body = self.alternative || make_node(AST_EmptyStatement, self);
        self.alternative = tmp;
    }
    if (is_empty(self.body) && is_empty(self.alternative)) {
        return make_node(AST_SimpleStatement, self.condition, {
            body: self.condition.clone()
        }).optimize(compressor);
    }
    if (self.body instanceof AST_SimpleStatement
        && self.alternative instanceof AST_SimpleStatement) {
        return make_node(AST_SimpleStatement, self, {
            body: make_node(AST_Conditional, self, {
                condition   : self.condition,
                consequent  : self.body.body,
                alternative : self.alternative.body
            })
        }).optimize(compressor);
    }
    if (is_empty(self.alternative) && self.body instanceof AST_SimpleStatement) {
        if (self_condition_length === negated_length && !negated_is_best
            && self.condition instanceof AST_Binary && self.condition.operator == "||") {
            // although the code length of self.condition and negated are the same,
            // negated does not require additional surrounding parentheses.
            // see https://github.com/mishoo/UglifyJS2/issues/979
            negated_is_best = true;
        }
        if (negated_is_best) return make_node(AST_SimpleStatement, self, {
            body: make_node(AST_Binary, self, {
                operator : "||",
                left     : negated,
                right    : self.body.body
            })
        }).optimize(compressor);
        return make_node(AST_SimpleStatement, self, {
            body: make_node(AST_Binary, self, {
                operator : "&&",
                left     : self.condition,
                right    : self.body.body
            })
        }).optimize(compressor);
    }
    if (self.body instanceof AST_EmptyStatement
        && self.alternative instanceof AST_SimpleStatement) {
        return make_node(AST_SimpleStatement, self, {
            body: make_node(AST_Binary, self, {
                operator : "||",
                left     : self.condition,
                right    : self.alternative.body
            })
        }).optimize(compressor);
    }
    if (self.body instanceof AST_Exit
        && self.alternative instanceof AST_Exit
        && self.body.TYPE == self.alternative.TYPE) {
        return make_node(self.body.CTOR, self, {
            value: make_node(AST_Conditional, self, {
                condition   : self.condition,
                consequent  : self.body.value || make_node(AST_Undefined, self.body),
                alternative : self.alternative.value || make_node(AST_Undefined, self.alternative)
            }).transform(compressor)
        }).optimize(compressor);
    }
    if (self.body instanceof AST_If
        && !self.body.alternative
        && !self.alternative) {
        self = make_node(AST_If, self, {
            condition: make_node(AST_Binary, self.condition, {
                operator: "&&",
                left: self.condition,
                right: self.body.condition
            }),
            body: self.body.body,
            alternative: null
        });
    }
    if (aborts(self.body)) {
        if (self.alternative) {
            var alt = self.alternative;
            self.alternative = null;
            return make_node(AST_BlockStatement, self, {
                body: [ self, alt ]
            }).optimize(compressor);
        }
    }
    if (aborts(self.alternative)) {
        var body = self.body;
        self.body = self.alternative;
        self.condition = negated_is_best ? negated : self.condition.negate(compressor);
        self.alternative = null;
        return make_node(AST_BlockStatement, self, {
            body: [ self, body ]
        }).optimize(compressor);
    }
    return self;
});

def_optimize(AST_Switch, function(self, compressor) {
    if (!compressor.option("switches")) return self;
    var branch;
    var value = self.expression.evaluate(compressor);
    if (!(value instanceof AST_Node)) {
        var orig = self.expression;
        self.expression = make_node_from_constant(value, orig);
        self.expression = best_of_expression(self.expression.transform(compressor), orig);
    }
    if (!compressor.option("dead_code")) return self;
    if (value instanceof AST_Node) {
        value = self.expression.tail_node().evaluate(compressor);
    }
    var decl = [];
    var body = [];
    var default_branch;
    var exact_match;
    for (var i = 0, len = self.body.length; i < len && !exact_match; i++) {
        branch = self.body[i];
        if (branch instanceof AST_Default) {
            if (!default_branch) {
                default_branch = branch;
            } else {
                eliminate_branch(branch, body[body.length - 1]);
            }
        } else if (!(value instanceof AST_Node)) {
            var exp = branch.expression.evaluate(compressor);
            if (!(exp instanceof AST_Node) && exp !== value) {
                eliminate_branch(branch, body[body.length - 1]);
                continue;
            }
            if (exp instanceof AST_Node) exp = branch.expression.tail_node().evaluate(compressor);
            if (exp === value) {
                exact_match = branch;
                if (default_branch) {
                    var default_index = body.indexOf(default_branch);
                    body.splice(default_index, 1);
                    eliminate_branch(default_branch, body[default_index - 1]);
                    default_branch = null;
                }
            }
        }
        if (aborts(branch)) {
            var prev = body[body.length - 1];
            if (aborts(prev) && prev.body.length == branch.body.length
                && make_node(AST_BlockStatement, prev, prev).equivalent_to(make_node(AST_BlockStatement, branch, branch))) {
                prev.body = [];
            }
        }
        body.push(branch);
    }
    while (i < len) eliminate_branch(self.body[i++], body[body.length - 1]);
    if (body.length > 0) {
        body[0].body = decl.concat(body[0].body);
    }
    self.body = body;
    while (branch = body[body.length - 1]) {
        var stat = branch.body[branch.body.length - 1];
        if (stat instanceof AST_Break && compressor.loopcontrol_target(stat) === self)
            branch.body.pop();
        if (branch.body.length || branch instanceof AST_Case
            && (default_branch || branch.expression.has_side_effects(compressor))) break;
        if (body.pop() === default_branch) default_branch = null;
    }
    if (body.length == 0) {
        return make_node(AST_BlockStatement, self, {
            body: decl.concat(make_node(AST_SimpleStatement, self.expression, {
                body: self.expression
            }))
        }).optimize(compressor);
    }
    if (body.length == 1 && (body[0] === exact_match || body[0] === default_branch)) {
        var has_break = false;
        var tw = new TreeWalker(function(node) {
            if (has_break
                || node instanceof AST_Lambda
                || node instanceof AST_SimpleStatement) return true;
            if (node instanceof AST_Break && tw.loopcontrol_target(node) === self)
                has_break = true;
        });
        self.walk(tw);
        if (!has_break) {
            var statements = body[0].body.slice();
            var exp = body[0].expression;
            if (exp) statements.unshift(make_node(AST_SimpleStatement, exp, {
                body: exp
            }));
            statements.unshift(make_node(AST_SimpleStatement, self.expression, {
                body:self.expression
            }));
            return make_node(AST_BlockStatement, self, {
                body: statements
            }).optimize(compressor);
        }
    }
    return self;

    function eliminate_branch(branch, prev) {
        if (prev && !aborts(prev)) {
            prev.body = prev.body.concat(branch.body);
        } else {
            trim_unreachable_code(compressor, branch, decl);
        }
    }
});

def_optimize(AST_Try, function(self, compressor) {
    tighten_body(self.body, compressor);
    if (self.bcatch && self.bfinally && self.bfinally.body.every(is_empty)) self.bfinally = null;
    if (compressor.option("dead_code") && self.body.every(is_empty)) {
        var body = [];
        if (self.bcatch) {
            trim_unreachable_code(compressor, self.bcatch, body);
        }
        if (self.bfinally) body.push(...self.bfinally.body);
        return make_node(AST_BlockStatement, self, {
            body: body
        }).optimize(compressor);
    }
    return self;
});

AST_Definitions.DEFMETHOD("remove_initializers", function() {
    var decls = [];
    this.definitions.forEach(function(def) {
        if (def.name instanceof AST_SymbolDeclaration) {
            def.value = null;
            decls.push(def);
        } else {
            walk(def.name, node => {
                if (node instanceof AST_SymbolDeclaration) {
                    decls.push(make_node(AST_VarDef, def, {
                        name: node,
                        value: null
                    }));
                }
            });
        }
    });
    this.definitions = decls;
});

AST_Definitions.DEFMETHOD("to_assignments", function(compressor) {
    var reduce_vars = compressor.option("reduce_vars");
    var assignments = [];

    for (const def of this.definitions) {
        if (def.value) {
            var name = make_node(AST_SymbolRef, def.name, def.name);
            assignments.push(make_node(AST_Assign, def, {
                operator : "=",
                logical: false,
                left     : name,
                right    : def.value
            }));
            if (reduce_vars) name.definition().fixed = false;
        } else if (def.value) {
            // Because it's a destructuring, do not turn into an assignment.
            var varDef = make_node(AST_VarDef, def, {
                name: def.name,
                value: def.value
            });
            var var_ = make_node(AST_Var, def, {
                definitions: [ varDef ]
            });
            assignments.push(var_);
        }
        const thedef = def.name.definition();
        thedef.eliminated++;
        thedef.replaced--;
    }

    if (assignments.length == 0) return null;
    return make_sequence(this, assignments);
});

def_optimize(AST_Definitions, function(self) {
    if (self.definitions.length == 0)
        return make_node(AST_EmptyStatement, self);
    return self;
});

def_optimize(AST_VarDef, function(self) {
    if (
        self.name instanceof AST_SymbolLet
        && self.value != null
        && is_undefined(self.value)
    ) {
        self.value = null;
    }
    return self;
});

def_optimize(AST_Import, function(self) {
    return self;
});

// TODO this only works with AST_Defun, shouldn't it work for other ways of defining functions?
function retain_top_func(fn, compressor) {
    return compressor.top_retain
        && fn instanceof AST_Defun
        && has_flag(fn, TOP)
        && fn.name
        && compressor.top_retain(fn.name);
}

def_optimize(AST_Call, function(self, compressor) {
    var exp = self.expression;
    var fn = exp;
    inline_array_like_spread(self.args);
    var simple_args = self.args.every((arg) =>
        !(arg instanceof AST_Expansion)
    );

    if (compressor.option("reduce_vars")
        && fn instanceof AST_SymbolRef
        && !has_annotation(self, _NOINLINE)
    ) {
        const fixed = fn.fixed_value();
        if (!retain_top_func(fixed, compressor)) {
            fn = fixed;
        }
    }

    if (self.optional && is_nullish(fn)) {
        return make_node(AST_Undefined, self);
    }

    var is_func = fn instanceof AST_Lambda;

    if (is_func && fn.pinned()) return self;

    if (compressor.option("unused")
        && simple_args
        && is_func
        && !fn.uses_arguments) {
        var pos = 0, last = 0;
        for (var i = 0, len = self.args.length; i < len; i++) {
            if (fn.argnames[i] instanceof AST_Expansion) {
                if (has_flag(fn.argnames[i].expression, UNUSED)) while (i < len) {
                    var node = self.args[i++].drop_side_effect_free(compressor);
                    if (node) {
                        self.args[pos++] = node;
                    }
                } else while (i < len) {
                    self.args[pos++] = self.args[i++];
                }
                last = pos;
                break;
            }
            var trim = i >= fn.argnames.length;
            if (trim || has_flag(fn.argnames[i], UNUSED)) {
                var node = self.args[i].drop_side_effect_free(compressor);
                if (node) {
                    self.args[pos++] = node;
                } else if (!trim) {
                    self.args[pos++] = make_node(AST_Number, self.args[i], {
                        value: 0
                    });
                    continue;
                }
            } else {
                self.args[pos++] = self.args[i];
            }
            last = pos;
        }
        self.args.length = last;
    }

    if (compressor.option("unsafe")) {
        if (is_undeclared_ref(exp)) switch (exp.name) {
          case "Array":
            if (self.args.length != 1) {
                return make_node(AST_Array, self, {
                    elements: self.args
                }).optimize(compressor);
            } else if (self.args[0] instanceof AST_Number && self.args[0].value <= 11) {
                const elements = [];
                for (let i = 0; i < self.args[0].value; i++) elements.push(new AST_Hole);
                return new AST_Array({ elements });
            }
            break;
          case "Object":
            if (self.args.length == 0) {
                return make_node(AST_Object, self, {
                    properties: []
                });
            }
            break;
          case "String":
            if (self.args.length == 0) return make_node(AST_String, self, {
                value: ""
            });
            if (self.args.length <= 1) return make_node(AST_Binary, self, {
                left: self.args[0],
                operator: "+",
                right: make_node(AST_String, self, { value: "" })
            }).optimize(compressor);
            break;
          case "Number":
            if (self.args.length == 0) return make_node(AST_Number, self, {
                value: 0
            });
            if (self.args.length == 1 && compressor.option("unsafe_math")) {
                return make_node(AST_UnaryPrefix, self, {
                    expression: self.args[0],
                    operator: "+"
                }).optimize(compressor);
            }
            break;
          case "Symbol":
            if (self.args.length == 1 && self.args[0] instanceof AST_String && compressor.option("unsafe_symbols"))
                self.args.length = 0;
                break;
          case "Boolean":
            if (self.args.length == 0) return make_node(AST_False, self);
            if (self.args.length == 1) return make_node(AST_UnaryPrefix, self, {
                expression: make_node(AST_UnaryPrefix, self, {
                    expression: self.args[0],
                    operator: "!"
                }),
                operator: "!"
            }).optimize(compressor);
            break;
          case "RegExp":
            var params = [];
            if (self.args.length >= 1
                && self.args.length <= 2
                && self.args.every((arg) => {
                    var value = arg.evaluate(compressor);
                    params.push(value);
                    return arg !== value;
                })
            ) {
                let [ source, flags ] = params;
                source = regexp_source_fix(new RegExp(source).source);
                const rx = make_node(AST_RegExp, self, {
                    value: { source, flags }
                });
                if (rx._eval(compressor) !== rx) {
                    return rx;
                }
            }
            break;
        } else if (exp instanceof AST_Dot) switch(exp.property) {
          case "toString":
            if (self.args.length == 0 && !exp.expression.may_throw_on_access(compressor)) {
                return make_node(AST_Binary, self, {
                    left: make_node(AST_String, self, { value: "" }),
                    operator: "+",
                    right: exp.expression
                }).optimize(compressor);
            }
            break;
          case "join":
            if (exp.expression instanceof AST_Array) EXIT: {
                var separator;
                if (self.args.length > 0) {
                    separator = self.args[0].evaluate(compressor);
                    if (separator === self.args[0]) break EXIT; // not a constant
                }
                var elements = [];
                var consts = [];
                for (var i = 0, len = exp.expression.elements.length; i < len; i++) {
                    var el = exp.expression.elements[i];
                    if (el instanceof AST_Expansion) break EXIT;
                    var value = el.evaluate(compressor);
                    if (value !== el) {
                        consts.push(value);
                    } else {
                        if (consts.length > 0) {
                            elements.push(make_node(AST_String, self, {
                                value: consts.join(separator)
                            }));
                            consts.length = 0;
                        }
                        elements.push(el);
                    }
                }
                if (consts.length > 0) {
                    elements.push(make_node(AST_String, self, {
                        value: consts.join(separator)
                    }));
                }
                if (elements.length == 0) return make_node(AST_String, self, { value: "" });
                if (elements.length == 1) {
                    if (elements[0].is_string(compressor)) {
                        return elements[0];
                    }
                    return make_node(AST_Binary, elements[0], {
                        operator : "+",
                        left     : make_node(AST_String, self, { value: "" }),
                        right    : elements[0]
                    });
                }
                if (separator == "") {
                    var first;
                    if (elements[0].is_string(compressor)
                        || elements[1].is_string(compressor)) {
                        first = elements.shift();
                    } else {
                        first = make_node(AST_String, self, { value: "" });
                    }
                    return elements.reduce(function(prev, el) {
                        return make_node(AST_Binary, el, {
                            operator : "+",
                            left     : prev,
                            right    : el
                        });
                    }, first).optimize(compressor);
                }
                // need this awkward cloning to not affect original element
                // best_of will decide which one to get through.
                var node = self.clone();
                node.expression = node.expression.clone();
                node.expression.expression = node.expression.expression.clone();
                node.expression.expression.elements = elements;
                return best_of(compressor, self, node);
            }
            break;
          case "charAt":
            if (exp.expression.is_string(compressor)) {
                var arg = self.args[0];
                var index = arg ? arg.evaluate(compressor) : 0;
                if (index !== arg) {
                    return make_node(AST_Sub, exp, {
                        expression: exp.expression,
                        property: make_node_from_constant(index | 0, arg || exp)
                    }).optimize(compressor);
                }
            }
            break;
          case "apply":
            if (self.args.length == 2 && self.args[1] instanceof AST_Array) {
                var args = self.args[1].elements.slice();
                args.unshift(self.args[0]);
                return make_node(AST_Call, self, {
                    expression: make_node(AST_Dot, exp, {
                        expression: exp.expression,
                        optional: false,
                        property: "call"
                    }),
                    args: args
                }).optimize(compressor);
            }
            break;
          case "call":
            var func = exp.expression;
            if (func instanceof AST_SymbolRef) {
                func = func.fixed_value();
            }
            if (func instanceof AST_Lambda && !func.contains_this()) {
                return (self.args.length ? make_sequence(this, [
                    self.args[0],
                    make_node(AST_Call, self, {
                        expression: exp.expression,
                        args: self.args.slice(1)
                    })
                ]) : make_node(AST_Call, self, {
                    expression: exp.expression,
                    args: []
                })).optimize(compressor);
            }
            break;
        }
    }

    if (compressor.option("unsafe_Function")
        && is_undeclared_ref(exp)
        && exp.name == "Function") {
        // new Function() => function(){}
        if (self.args.length == 0) return make_node(AST_Function, self, {
            argnames: [],
            body: []
        }).optimize(compressor);
        if (self.args.every((x) => x instanceof AST_String)) {
            // quite a corner-case, but we can handle it:
            //   https://github.com/mishoo/UglifyJS2/issues/203
            // if the code argument is a constant, then we can minify it.
            try {
                var code = "n(function(" + self.args.slice(0, -1).map(function(arg) {
                    return arg.value;
                }).join(",") + "){" + self.args[self.args.length - 1].value + "})";
                var ast = parse(code);
                var mangle = { ie8: compressor.option("ie8") };
                ast.figure_out_scope(mangle);
                var comp = new Compressor(compressor.options, {
                    mangle_options: compressor.mangle_options
                });
                ast = ast.transform(comp);
                ast.figure_out_scope(mangle);
                base54.reset();
                ast.compute_char_frequency(mangle);
                ast.mangle_names(mangle);
                var fun;
                walk(ast, node => {
                    if (is_func_expr(node)) {
                        fun = node;
                        return walk_abort;
                    }
                });
                var code = OutputStream();
                AST_BlockStatement.prototype._codegen.call(fun, fun, code);
                self.args = [
                    make_node(AST_String, self, {
                        value: fun.argnames.map(function(arg) {
                            return arg.print_to_string();
                        }).join(",")
                    }),
                    make_node(AST_String, self.args[self.args.length - 1], {
                        value: code.get().replace(/^{|}$/g, "")
                    })
                ];
                return self;
            } catch (ex) {
                if (!(ex instanceof JS_Parse_Error)) {
                    throw ex;
                }

                // Otherwise, it crashes at runtime. Or maybe it's nonstandard syntax.
            }
        }
    }

    var stat = is_func && fn.body[0];
    var is_regular_func = is_func && !fn.is_generator && !fn.async;
    var can_inline = is_regular_func && compressor.option("inline") && !self.is_callee_pure(compressor);
    if (can_inline && stat instanceof AST_Return) {
        let returned = stat.value;
        if (!returned || returned.is_constant_expression()) {
            if (returned) {
                returned = returned.clone(true);
            } else {
                returned = make_node(AST_Undefined, self);
            }
            const args = self.args.concat(returned);
            return make_sequence(self, args).optimize(compressor);
        }

        // optimize identity function
        if (
            fn.argnames.length === 1
            && (fn.argnames[0] instanceof AST_SymbolFunarg)
            && self.args.length < 2
            && returned instanceof AST_SymbolRef
            && returned.name === fn.argnames[0].name
        ) {
            const replacement =
                (self.args[0] || make_node(AST_Undefined)).optimize(compressor);

            let parent;
            if (
                replacement instanceof AST_PropAccess
                && (parent = compressor.parent()) instanceof AST_Call
                && parent.expression === self
            ) {
                // identity function was being used to remove `this`, like in
                //
                // id(bag.no_this)(...)
                //
                // Replace with a larger but more effish (0, bag.no_this) wrapper.

                return make_sequence(self, [
                    make_node(AST_Number, self, { value: 0 }),
                    replacement
                ]);
            }
            // replace call with first argument or undefined if none passed
            return replacement;
        }
    }

    if (can_inline) {
        var scope, in_loop, level = -1;
        let def;
        let returned_value;
        let nearest_scope;
        if (simple_args
            && !fn.uses_arguments
            && !(compressor.parent() instanceof AST_Class)
            && !(fn.name && fn instanceof AST_Function)
            && (returned_value = can_flatten_body(stat))
            && (exp === fn
                || has_annotation(self, _INLINE)
                || compressor.option("unused")
                    && (def = exp.definition()).references.length == 1
                    && !recursive_ref(compressor, def)
                    && fn.is_constant_expression(exp.scope))
            && !has_annotation(self, _PURE | _NOINLINE)
            && !fn.contains_this()
            && can_inject_symbols()
            && (nearest_scope = find_scope(compressor))
            && !scope_encloses_variables_in_this_scope(nearest_scope, fn)
            && !(function in_default_assign() {
                    // Due to the fact function parameters have their own scope
                    // which can't use `var something` in the function body within,
                    // we simply don't inline into DefaultAssign.
                    let i = 0;
                    let p;
                    while ((p = compressor.parent(i++))) {
                        if (p instanceof AST_DefaultAssign) return true;
                        if (p instanceof AST_Block) break;
                    }
                    return false;
                })()
            && !(scope instanceof AST_Class)
        ) {
            set_flag(fn, SQUEEZED);
            nearest_scope.add_child_scope(fn);
            return make_sequence(self, flatten_fn(returned_value)).optimize(compressor);
        }
    }

    if (can_inline && has_annotation(self, _INLINE)) {
        set_flag(fn, SQUEEZED);
        fn = make_node(fn.CTOR === AST_Defun ? AST_Function : fn.CTOR, fn, fn);
        fn.figure_out_scope({}, {
            parent_scope: find_scope(compressor),
            toplevel: compressor.get_toplevel()
        });

        return make_node(AST_Call, self, {
            expression: fn,
            args: self.args,
        }).optimize(compressor);
    }

    const can_drop_this_call = is_regular_func && compressor.option("side_effects") && fn.body.every(is_empty);
    if (can_drop_this_call) {
        var args = self.args.concat(make_node(AST_Undefined, self));
        return make_sequence(self, args).optimize(compressor);
    }

    if (compressor.option("negate_iife")
        && compressor.parent() instanceof AST_SimpleStatement
        && is_iife_call(self)) {
        return self.negate(compressor, true);
    }

    var ev = self.evaluate(compressor);
    if (ev !== self) {
        ev = make_node_from_constant(ev, self).optimize(compressor);
        return best_of(compressor, ev, self);
    }

    return self;

    function return_value(stat) {
        if (!stat) return make_node(AST_Undefined, self);
        if (stat instanceof AST_Return) {
            if (!stat.value) return make_node(AST_Undefined, self);
            return stat.value.clone(true);
        }
        if (stat instanceof AST_SimpleStatement) {
            return make_node(AST_UnaryPrefix, stat, {
                operator: "void",
                expression: stat.body.clone(true)
            });
        }
    }

    function can_flatten_body(stat) {
        var body = fn.body;
        var len = body.length;
        if (compressor.option("inline") < 3) {
            return len == 1 && return_value(stat);
        }
        stat = null;
        for (var i = 0; i < len; i++) {
            var line = body[i];
            if (line instanceof AST_Var) {
                if (stat && !line.definitions.every((var_def) =>
                    !var_def.value
                )) {
                    return false;
                }
            } else if (stat) {
                return false;
            } else if (!(line instanceof AST_EmptyStatement)) {
                stat = line;
            }
        }
        return return_value(stat);
    }

    function can_inject_args(block_scoped, safe_to_inject) {
        for (var i = 0, len = fn.argnames.length; i < len; i++) {
            var arg = fn.argnames[i];
            if (arg instanceof AST_DefaultAssign) {
                if (has_flag(arg.left, UNUSED)) continue;
                return false;
            }
            if (arg instanceof AST_Destructuring) return false;
            if (arg instanceof AST_Expansion) {
                if (has_flag(arg.expression, UNUSED)) continue;
                return false;
            }
            if (has_flag(arg, UNUSED)) continue;
            if (!safe_to_inject
                || block_scoped.has(arg.name)
                || identifier_atom.has(arg.name)
                || scope.conflicting_def(arg.name)) {
                return false;
            }
            if (in_loop) in_loop.push(arg.definition());
        }
        return true;
    }

    function can_inject_vars(block_scoped, safe_to_inject) {
        var len = fn.body.length;
        for (var i = 0; i < len; i++) {
            var stat = fn.body[i];
            if (!(stat instanceof AST_Var)) continue;
            if (!safe_to_inject) return false;
            for (var j = stat.definitions.length; --j >= 0;) {
                var name = stat.definitions[j].name;
                if (name instanceof AST_Destructuring
                    || block_scoped.has(name.name)
                    || identifier_atom.has(name.name)
                    || scope.conflicting_def(name.name)) {
                    return false;
                }
                if (in_loop) in_loop.push(name.definition());
            }
        }
        return true;
    }

    function can_inject_symbols() {
        var block_scoped = new Set();
        do {
            scope = compressor.parent(++level);
            if (scope.is_block_scope() && scope.block_scope) {
                // TODO this is sometimes undefined during compression.
                // But it should always have a value!
                scope.block_scope.variables.forEach(function (variable) {
                    block_scoped.add(variable.name);
                });
            }
            if (scope instanceof AST_Catch) {
                // TODO can we delete? AST_Catch is a block scope.
                if (scope.argname) {
                    block_scoped.add(scope.argname.name);
                }
            } else if (scope instanceof AST_IterationStatement) {
                in_loop = [];
            } else if (scope instanceof AST_SymbolRef) {
                if (scope.fixed_value() instanceof AST_Scope) return false;
            }
        } while (!(scope instanceof AST_Scope));

        var safe_to_inject = !(scope instanceof AST_Toplevel) || compressor.toplevel.vars;
        var inline = compressor.option("inline");
        if (!can_inject_vars(block_scoped, inline >= 3 && safe_to_inject)) return false;
        if (!can_inject_args(block_scoped, inline >= 2 && safe_to_inject)) return false;
        return !in_loop || in_loop.length == 0 || !is_reachable(fn, in_loop);
    }

    function append_var(decls, expressions, name, value) {
        var def = name.definition();

        // Name already exists, only when a function argument had the same name
        const already_appended = scope.variables.has(name.name);
        if (!already_appended) {
            scope.variables.set(name.name, def);
            scope.enclosed.push(def);
            decls.push(make_node(AST_VarDef, name, {
                name: name,
                value: null
            }));
        }

        var sym = make_node(AST_SymbolRef, name, name);
        def.references.push(sym);
        if (value) expressions.push(make_node(AST_Assign, self, {
            operator: "=",
            logical: false,
            left: sym,
            right: value.clone()
        }));
    }

    function flatten_args(decls, expressions) {
        var len = fn.argnames.length;
        for (var i = self.args.length; --i >= len;) {
            expressions.push(self.args[i]);
        }
        for (i = len; --i >= 0;) {
            var name = fn.argnames[i];
            var value = self.args[i];
            if (has_flag(name, UNUSED) || !name.name || scope.conflicting_def(name.name)) {
                if (value) expressions.push(value);
            } else {
                var symbol = make_node(AST_SymbolVar, name, name);
                name.definition().orig.push(symbol);
                if (!value && in_loop) value = make_node(AST_Undefined, self);
                append_var(decls, expressions, symbol, value);
            }
        }
        decls.reverse();
        expressions.reverse();
    }

    function flatten_vars(decls, expressions) {
        var pos = expressions.length;
        for (var i = 0, lines = fn.body.length; i < lines; i++) {
            var stat = fn.body[i];
            if (!(stat instanceof AST_Var)) continue;
            for (var j = 0, defs = stat.definitions.length; j < defs; j++) {
                var var_def = stat.definitions[j];
                var name = var_def.name;
                append_var(decls, expressions, name, var_def.value);
                if (in_loop && fn.argnames.every((argname) =>
                    argname.name != name.name
                )) {
                    var def = fn.variables.get(name.name);
                    var sym = make_node(AST_SymbolRef, name, name);
                    def.references.push(sym);
                    expressions.splice(pos++, 0, make_node(AST_Assign, var_def, {
                        operator: "=",
                        logical: false,
                        left: sym,
                        right: make_node(AST_Undefined, name)
                    }));
                }
            }
        }
    }

    function flatten_fn(returned_value) {
        var decls = [];
        var expressions = [];
        flatten_args(decls, expressions);
        flatten_vars(decls, expressions);
        expressions.push(returned_value);
        if (decls.length) {
            const i = scope.body.indexOf(compressor.parent(level - 1)) + 1;
            scope.body.splice(i, 0, make_node(AST_Var, fn, {
                definitions: decls
            }));
        }
        return expressions.map(exp => exp.clone(true));
    }
});

def_optimize(AST_New, function(self, compressor) {
    if (
        compressor.option("unsafe") &&
        is_undeclared_ref(self.expression) &&
        ["Object", "RegExp", "Function", "Error", "Array"].includes(self.expression.name)
    ) return make_node(AST_Call, self, self).transform(compressor);
    return self;
});

def_optimize(AST_Sequence, function(self, compressor) {
    if (!compressor.option("side_effects")) return self;
    var expressions = [];
    filter_for_side_effects();
    var end = expressions.length - 1;
    trim_right_for_undefined();
    if (end == 0) {
        self = maintain_this_binding(compressor.parent(), compressor.self(), expressions[0]);
        if (!(self instanceof AST_Sequence)) self = self.optimize(compressor);
        return self;
    }
    self.expressions = expressions;
    return self;

    function filter_for_side_effects() {
        var first = first_in_statement(compressor);
        var last = self.expressions.length - 1;
        self.expressions.forEach(function(expr, index) {
            if (index < last) expr = expr.drop_side_effect_free(compressor, first);
            if (expr) {
                merge_sequence(expressions, expr);
                first = false;
            }
        });
    }

    function trim_right_for_undefined() {
        while (end > 0 && is_undefined(expressions[end], compressor)) end--;
        if (end < expressions.length - 1) {
            expressions[end] = make_node(AST_UnaryPrefix, self, {
                operator   : "void",
                expression : expressions[end]
            });
            expressions.length = end + 1;
        }
    }
});

AST_Unary.DEFMETHOD("lift_sequences", function(compressor) {
    if (compressor.option("sequences")) {
        if (this.expression instanceof AST_Sequence) {
            var x = this.expression.expressions.slice();
            var e = this.clone();
            e.expression = x.pop();
            x.push(e);
            return make_sequence(this, x).optimize(compressor);
        }
    }
    return this;
});

def_optimize(AST_UnaryPostfix, function(self, compressor) {
    return self.lift_sequences(compressor);
});

def_optimize(AST_UnaryPrefix, function(self, compressor) {
    var e = self.expression;
    if (self.operator == "delete"
        && !(e instanceof AST_SymbolRef
            || e instanceof AST_PropAccess
            || is_identifier_atom(e))) {
        if (e instanceof AST_Sequence) {
            const exprs = e.expressions.slice();
            exprs.push(make_node(AST_True, self));
            return make_sequence(self, exprs).optimize(compressor);
        }
        return make_sequence(self, [ e, make_node(AST_True, self) ]).optimize(compressor);
    }
    var seq = self.lift_sequences(compressor);
    if (seq !== self) {
        return seq;
    }
    if (compressor.option("side_effects") && self.operator == "void") {
        e = e.drop_side_effect_free(compressor);
        if (e) {
            self.expression = e;
            return self;
        } else {
            return make_node(AST_Undefined, self).optimize(compressor);
        }
    }
    if (compressor.in_boolean_context()) {
        switch (self.operator) {
          case "!":
            if (e instanceof AST_UnaryPrefix && e.operator == "!") {
                // !!foo ==> foo, if we're in boolean context
                return e.expression;
            }
            if (e instanceof AST_Binary) {
                self = best_of(compressor, self, e.negate(compressor, first_in_statement(compressor)));
            }
            break;
          case "typeof":
            // typeof always returns a non-empty string, thus it's
            // always true in booleans
            // And we don't need to check if it's undeclared, because in typeof, that's OK
            return (e instanceof AST_SymbolRef ? make_node(AST_True, self) : make_sequence(self, [
                e,
                make_node(AST_True, self)
            ])).optimize(compressor);
        }
    }
    if (self.operator == "-" && e instanceof AST_Infinity) {
        e = e.transform(compressor);
    }
    if (e instanceof AST_Binary
        && (self.operator == "+" || self.operator == "-")
        && (e.operator == "*" || e.operator == "/" || e.operator == "%")) {
        return make_node(AST_Binary, self, {
            operator: e.operator,
            left: make_node(AST_UnaryPrefix, e.left, {
                operator: self.operator,
                expression: e.left
            }),
            right: e.right
        });
    }
    // avoids infinite recursion of numerals
    if (self.operator != "-"
        || !(e instanceof AST_Number || e instanceof AST_Infinity || e instanceof AST_BigInt)) {
        var ev = self.evaluate(compressor);
        if (ev !== self) {
            ev = make_node_from_constant(ev, self).optimize(compressor);
            return best_of(compressor, ev, self);
        }
    }
    return self;
});

AST_Binary.DEFMETHOD("lift_sequences", function(compressor) {
    if (compressor.option("sequences")) {
        if (this.left instanceof AST_Sequence) {
            var x = this.left.expressions.slice();
            var e = this.clone();
            e.left = x.pop();
            x.push(e);
            return make_sequence(this, x).optimize(compressor);
        }
        if (this.right instanceof AST_Sequence && !this.left.has_side_effects(compressor)) {
            var assign = this.operator == "=" && this.left instanceof AST_SymbolRef;
            var x = this.right.expressions;
            var last = x.length - 1;
            for (var i = 0; i < last; i++) {
                if (!assign && x[i].has_side_effects(compressor)) break;
            }
            if (i == last) {
                x = x.slice();
                var e = this.clone();
                e.right = x.pop();
                x.push(e);
                return make_sequence(this, x).optimize(compressor);
            } else if (i > 0) {
                var e = this.clone();
                e.right = make_sequence(this.right, x.slice(i));
                x = x.slice(0, i);
                x.push(e);
                return make_sequence(this, x).optimize(compressor);
            }
        }
    }
    return this;
});

var commutativeOperators = makePredicate("== === != !== * & | ^");
function is_object(node) {
    return node instanceof AST_Array
        || node instanceof AST_Lambda
        || node instanceof AST_Object
        || node instanceof AST_Class;
}

def_optimize(AST_Binary, function(self, compressor) {
    function reversible() {
        return self.left.is_constant()
            || self.right.is_constant()
            || !self.left.has_side_effects(compressor)
                && !self.right.has_side_effects(compressor);
    }
    function reverse(op) {
        if (reversible()) {
            if (op) self.operator = op;
            var tmp = self.left;
            self.left = self.right;
            self.right = tmp;
        }
    }
    if (commutativeOperators.has(self.operator)) {
        if (self.right.is_constant()
            && !self.left.is_constant()) {
            // if right is a constant, whatever side effects the
            // left side might have could not influence the
            // result.  hence, force switch.

            if (!(self.left instanceof AST_Binary
                  && PRECEDENCE[self.left.operator] >= PRECEDENCE[self.operator])) {
                reverse();
            }
        }
    }
    self = self.lift_sequences(compressor);
    if (compressor.option("comparisons")) switch (self.operator) {
      case "===":
      case "!==":
        var is_strict_comparison = true;
        if ((self.left.is_string(compressor) && self.right.is_string(compressor)) ||
            (self.left.is_number(compressor) && self.right.is_number(compressor)) ||
            (self.left.is_boolean() && self.right.is_boolean()) ||
            self.left.equivalent_to(self.right)) {
            self.operator = self.operator.substr(0, 2);
        }
        // XXX: intentionally falling down to the next case
      case "==":
      case "!=":
        // void 0 == x => null == x
        if (!is_strict_comparison && is_undefined(self.left, compressor)) {
            self.left = make_node(AST_Null, self.left);
        } else if (compressor.option("typeofs")
            // "undefined" == typeof x => undefined === x
            && self.left instanceof AST_String
            && self.left.value == "undefined"
            && self.right instanceof AST_UnaryPrefix
            && self.right.operator == "typeof") {
            var expr = self.right.expression;
            if (expr instanceof AST_SymbolRef ? expr.is_declared(compressor)
                : !(expr instanceof AST_PropAccess && compressor.option("ie8"))) {
                self.right = expr;
                self.left = make_node(AST_Undefined, self.left).optimize(compressor);
                if (self.operator.length == 2) self.operator += "=";
            }
        } else if (self.left instanceof AST_SymbolRef
            // obj !== obj => false
            && self.right instanceof AST_SymbolRef
            && self.left.definition() === self.right.definition()
            && is_object(self.left.fixed_value())) {
            return make_node(self.operator[0] == "=" ? AST_True : AST_False, self);
        }
        break;
      case "&&":
      case "||":
        var lhs = self.left;
        if (lhs.operator == self.operator) {
            lhs = lhs.right;
        }
        if (lhs instanceof AST_Binary
            && lhs.operator == (self.operator == "&&" ? "!==" : "===")
            && self.right instanceof AST_Binary
            && lhs.operator == self.right.operator
            && (is_undefined(lhs.left, compressor) && self.right.left instanceof AST_Null
                || lhs.left instanceof AST_Null && is_undefined(self.right.left, compressor))
            && !lhs.right.has_side_effects(compressor)
            && lhs.right.equivalent_to(self.right.right)) {
            var combined = make_node(AST_Binary, self, {
                operator: lhs.operator.slice(0, -1),
                left: make_node(AST_Null, self),
                right: lhs.right
            });
            if (lhs !== self.left) {
                combined = make_node(AST_Binary, self, {
                    operator: self.operator,
                    left: self.left.left,
                    right: combined
                });
            }
            return combined;
        }
        break;
    }
    if (self.operator == "+" && compressor.in_boolean_context()) {
        var ll = self.left.evaluate(compressor);
        var rr = self.right.evaluate(compressor);
        if (ll && typeof ll == "string") {
            return make_sequence(self, [
                self.right,
                make_node(AST_True, self)
            ]).optimize(compressor);
        }
        if (rr && typeof rr == "string") {
            return make_sequence(self, [
                self.left,
                make_node(AST_True, self)
            ]).optimize(compressor);
        }
    }
    if (compressor.option("comparisons") && self.is_boolean()) {
        if (!(compressor.parent() instanceof AST_Binary)
            || compressor.parent() instanceof AST_Assign) {
            var negated = make_node(AST_UnaryPrefix, self, {
                operator: "!",
                expression: self.negate(compressor, first_in_statement(compressor))
            });
            self = best_of(compressor, self, negated);
        }
        if (compressor.option("unsafe_comps")) {
            switch (self.operator) {
              case "<": reverse(">"); break;
              case "<=": reverse(">="); break;
            }
        }
    }
    if (self.operator == "+") {
        if (self.right instanceof AST_String
            && self.right.getValue() == ""
            && self.left.is_string(compressor)) {
            return self.left;
        }
        if (self.left instanceof AST_String
            && self.left.getValue() == ""
            && self.right.is_string(compressor)) {
            return self.right;
        }
        if (self.left instanceof AST_Binary
            && self.left.operator == "+"
            && self.left.left instanceof AST_String
            && self.left.left.getValue() == ""
            && self.right.is_string(compressor)) {
            self.left = self.left.right;
            return self;
        }
    }
    if (compressor.option("evaluate")) {
        switch (self.operator) {
          case "&&":
            var ll = has_flag(self.left, TRUTHY)
                ? true
                : has_flag(self.left, FALSY)
                    ? false
                    : self.left.evaluate(compressor);
            if (!ll) {
                return maintain_this_binding(compressor.parent(), compressor.self(), self.left).optimize(compressor);
            } else if (!(ll instanceof AST_Node)) {
                return make_sequence(self, [ self.left, self.right ]).optimize(compressor);
            }
            var rr = self.right.evaluate(compressor);
            if (!rr) {
                if (compressor.in_boolean_context()) {
                    return make_sequence(self, [
                        self.left,
                        make_node(AST_False, self)
                    ]).optimize(compressor);
                } else {
                    set_flag(self, FALSY);
                }
            } else if (!(rr instanceof AST_Node)) {
                var parent = compressor.parent();
                if (parent.operator == "&&" && parent.left === compressor.self() || compressor.in_boolean_context()) {
                    return self.left.optimize(compressor);
                }
            }
            // x || false && y ---> x ? y : false
            if (self.left.operator == "||") {
                var lr = self.left.right.evaluate(compressor);
                if (!lr) return make_node(AST_Conditional, self, {
                    condition: self.left.left,
                    consequent: self.right,
                    alternative: self.left.right
                }).optimize(compressor);
            }
            break;
          case "||":
            var ll = has_flag(self.left, TRUTHY)
              ? true
              : has_flag(self.left, FALSY)
                ? false
                : self.left.evaluate(compressor);
            if (!ll) {
                return make_sequence(self, [ self.left, self.right ]).optimize(compressor);
            } else if (!(ll instanceof AST_Node)) {
                return maintain_this_binding(compressor.parent(), compressor.self(), self.left).optimize(compressor);
            }
            var rr = self.right.evaluate(compressor);
            if (!rr) {
                var parent = compressor.parent();
                if (parent.operator == "||" && parent.left === compressor.self() || compressor.in_boolean_context()) {
                    return self.left.optimize(compressor);
                }
            } else if (!(rr instanceof AST_Node)) {
                if (compressor.in_boolean_context()) {
                    return make_sequence(self, [
                        self.left,
                        make_node(AST_True, self)
                    ]).optimize(compressor);
                } else {
                    set_flag(self, TRUTHY);
                }
            }
            if (self.left.operator == "&&") {
                var lr = self.left.right.evaluate(compressor);
                if (lr && !(lr instanceof AST_Node)) return make_node(AST_Conditional, self, {
                    condition: self.left.left,
                    consequent: self.left.right,
                    alternative: self.right
                }).optimize(compressor);
            }
            break;
          case "??":
            if (is_nullish(self.left)) {
                return self.right;
            }

            var ll = self.left.evaluate(compressor);
            if (!(ll instanceof AST_Node)) {
                // if we know the value for sure we can simply compute right away.
                return ll == null ? self.right : self.left;
            }

            if (compressor.in_boolean_context()) {
                const rr = self.right.evaluate(compressor);
                if (!(rr instanceof AST_Node) && !rr) {
                    return self.left;
                }
            }
        }
        var associative = true;
        switch (self.operator) {
          case "+":
            // (x + "foo") + "bar" => x + "foobar"
            if (self.right instanceof AST_Constant
                && self.left instanceof AST_Binary
                && self.left.operator == "+"
                && self.left.is_string(compressor)) {
                var binary = make_node(AST_Binary, self, {
                    operator: "+",
                    left: self.left.right,
                    right: self.right,
                });
                var r = binary.optimize(compressor);
                if (binary !== r) {
                    self = make_node(AST_Binary, self, {
                        operator: "+",
                        left: self.left.left,
                        right: r
                    });
                }
            }
            // (x + "foo") + ("bar" + y) => (x + "foobar") + y
            if (self.left instanceof AST_Binary
                && self.left.operator == "+"
                && self.left.is_string(compressor)
                && self.right instanceof AST_Binary
                && self.right.operator == "+"
                && self.right.is_string(compressor)) {
                var binary = make_node(AST_Binary, self, {
                    operator: "+",
                    left: self.left.right,
                    right: self.right.left,
                });
                var m = binary.optimize(compressor);
                if (binary !== m) {
                    self = make_node(AST_Binary, self, {
                        operator: "+",
                        left: make_node(AST_Binary, self.left, {
                            operator: "+",
                            left: self.left.left,
                            right: m
                        }),
                        right: self.right.right
                    });
                }
            }
            // a + -b => a - b
            if (self.right instanceof AST_UnaryPrefix
                && self.right.operator == "-"
                && self.left.is_number(compressor)) {
                self = make_node(AST_Binary, self, {
                    operator: "-",
                    left: self.left,
                    right: self.right.expression
                });
                break;
            }
            // -a + b => b - a
            if (self.left instanceof AST_UnaryPrefix
                && self.left.operator == "-"
                && reversible()
                && self.right.is_number(compressor)) {
                self = make_node(AST_Binary, self, {
                    operator: "-",
                    left: self.right,
                    right: self.left.expression
                });
                break;
            }
            // `foo${bar}baz` + 1 => `foo${bar}baz1`
            if (self.left instanceof AST_TemplateString) {
                var l = self.left;
                var r = self.right.evaluate(compressor);
                if (r != self.right) {
                    l.segments[l.segments.length - 1].value += String(r);
                    return l;
                }
            }
            // 1 + `foo${bar}baz` => `1foo${bar}baz`
            if (self.right instanceof AST_TemplateString) {
                var r = self.right;
                var l = self.left.evaluate(compressor);
                if (l != self.left) {
                    r.segments[0].value = String(l) + r.segments[0].value;
                    return r;
                }
            }
            // `1${bar}2` + `foo${bar}baz` => `1${bar}2foo${bar}baz`
            if (self.left instanceof AST_TemplateString
                && self.right instanceof AST_TemplateString) {
                var l = self.left;
                var segments = l.segments;
                var r = self.right;
                segments[segments.length - 1].value += r.segments[0].value;
                for (var i = 1; i < r.segments.length; i++) {
                    segments.push(r.segments[i]);
                }
                return l;
            }
          case "*":
            associative = compressor.option("unsafe_math");
          case "&":
          case "|":
          case "^":
            // a + +b => +b + a
            if (self.left.is_number(compressor)
                && self.right.is_number(compressor)
                && reversible()
                && !(self.left instanceof AST_Binary
                    && self.left.operator != self.operator
                    && PRECEDENCE[self.left.operator] >= PRECEDENCE[self.operator])) {
                var reversed = make_node(AST_Binary, self, {
                    operator: self.operator,
                    left: self.right,
                    right: self.left
                });
                if (self.right instanceof AST_Constant
                    && !(self.left instanceof AST_Constant)) {
                    self = best_of(compressor, reversed, self);
                } else {
                    self = best_of(compressor, self, reversed);
                }
            }
            if (associative && self.is_number(compressor)) {
                // a + (b + c) => (a + b) + c
                if (self.right instanceof AST_Binary
                    && self.right.operator == self.operator) {
                    self = make_node(AST_Binary, self, {
                        operator: self.operator,
                        left: make_node(AST_Binary, self.left, {
                            operator: self.operator,
                            left: self.left,
                            right: self.right.left,
                            start: self.left.start,
                            end: self.right.left.end
                        }),
                        right: self.right.right
                    });
                }
                // (n + 2) + 3 => 5 + n
                // (2 * n) * 3 => 6 + n
                if (self.right instanceof AST_Constant
                    && self.left instanceof AST_Binary
                    && self.left.operator == self.operator) {
                    if (self.left.left instanceof AST_Constant) {
                        self = make_node(AST_Binary, self, {
                            operator: self.operator,
                            left: make_node(AST_Binary, self.left, {
                                operator: self.operator,
                                left: self.left.left,
                                right: self.right,
                                start: self.left.left.start,
                                end: self.right.end
                            }),
                            right: self.left.right
                        });
                    } else if (self.left.right instanceof AST_Constant) {
                        self = make_node(AST_Binary, self, {
                            operator: self.operator,
                            left: make_node(AST_Binary, self.left, {
                                operator: self.operator,
                                left: self.left.right,
                                right: self.right,
                                start: self.left.right.start,
                                end: self.right.end
                            }),
                            right: self.left.left
                        });
                    }
                }
                // (a | 1) | (2 | d) => (3 | a) | b
                if (self.left instanceof AST_Binary
                    && self.left.operator == self.operator
                    && self.left.right instanceof AST_Constant
                    && self.right instanceof AST_Binary
                    && self.right.operator == self.operator
                    && self.right.left instanceof AST_Constant) {
                    self = make_node(AST_Binary, self, {
                        operator: self.operator,
                        left: make_node(AST_Binary, self.left, {
                            operator: self.operator,
                            left: make_node(AST_Binary, self.left.left, {
                                operator: self.operator,
                                left: self.left.right,
                                right: self.right.left,
                                start: self.left.right.start,
                                end: self.right.left.end
                            }),
                            right: self.left.left
                        }),
                        right: self.right.right
                    });
                }
            }
        }
    }
    // x && (y && z)  ==>  x && y && z
    // x || (y || z)  ==>  x || y || z
    // x + ("y" + z)  ==>  x + "y" + z
    // "x" + (y + "z")==>  "x" + y + "z"
    if (self.right instanceof AST_Binary
        && self.right.operator == self.operator
        && (lazy_op.has(self.operator)
            || (self.operator == "+"
                && (self.right.left.is_string(compressor)
                    || (self.left.is_string(compressor)
                        && self.right.right.is_string(compressor)))))
    ) {
        self.left = make_node(AST_Binary, self.left, {
            operator : self.operator,
            left     : self.left.transform(compressor),
            right    : self.right.left.transform(compressor)
        });
        self.right = self.right.right.transform(compressor);
        return self.transform(compressor);
    }
    var ev = self.evaluate(compressor);
    if (ev !== self) {
        ev = make_node_from_constant(ev, self).optimize(compressor);
        return best_of(compressor, ev, self);
    }
    return self;
});

def_optimize(AST_SymbolExport, function(self) {
    return self;
});

function recursive_ref(compressor, def) {
    var node;
    for (var i = 0; node = compressor.parent(i); i++) {
        if (
            node instanceof AST_Lambda
            || node instanceof AST_Class
        ) {
            var name = node.name;
            if (name && name.definition() === def) break;
        }
    }
    return node;
}

function within_array_or_object_literal(compressor) {
    var node, level = 0;
    while (node = compressor.parent(level++)) {
        if (node instanceof AST_Statement) return false;
        if (node instanceof AST_Array
            || node instanceof AST_ObjectKeyVal
            || node instanceof AST_Object) {
            return true;
        }
    }
    return false;
}

def_optimize(AST_SymbolRef, function(self, compressor) {
    if (
        !compressor.option("ie8")
        && is_undeclared_ref(self)
        && !compressor.find_parent(AST_With)
    ) {
        switch (self.name) {
          case "undefined":
            return make_node(AST_Undefined, self).optimize(compressor);
          case "NaN":
            return make_node(AST_NaN, self).optimize(compressor);
          case "Infinity":
            return make_node(AST_Infinity, self).optimize(compressor);
        }
    }

    const parent = compressor.parent();
    if (compressor.option("reduce_vars") && is_lhs(self, parent) !== self) {
        const def = self.definition();
        const nearest_scope = find_scope(compressor);
        if (compressor.top_retain && def.global && compressor.top_retain(def)) {
            def.fixed = false;
            def.single_use = false;
            return self;
        }

        let fixed = self.fixed_value();
        let single_use = def.single_use
            && !(parent instanceof AST_Call
                && (parent.is_callee_pure(compressor))
                    || has_annotation(parent, _NOINLINE))
            && !(parent instanceof AST_Export
                && fixed instanceof AST_Lambda
                && fixed.name);

        if (single_use && fixed instanceof AST_Node) {
            single_use =
                !fixed.has_side_effects(compressor)
                && !fixed.may_throw(compressor);
        }

        if (single_use && (fixed instanceof AST_Lambda || fixed instanceof AST_Class)) {
            if (retain_top_func(fixed, compressor)) {
                single_use = false;
            } else if (def.scope !== self.scope
                && (def.escaped == 1
                    || has_flag(fixed, INLINED)
                    || within_array_or_object_literal(compressor)
                    || !compressor.option("reduce_funcs"))) {
                single_use = false;
            } else if (recursive_ref(compressor, def)) {
                single_use = false;
            } else if (def.scope !== self.scope || def.orig[0] instanceof AST_SymbolFunarg) {
                single_use = fixed.is_constant_expression(self.scope);
                if (single_use == "f") {
                    var scope = self.scope;
                    do {
                        if (scope instanceof AST_Defun || is_func_expr(scope)) {
                            set_flag(scope, INLINED);
                        }
                    } while (scope = scope.parent_scope);
                }
            }
        }

        if (single_use && fixed instanceof AST_Lambda) {
            single_use =
                def.scope === self.scope
                    && !scope_encloses_variables_in_this_scope(nearest_scope, fixed)
                || parent instanceof AST_Call
                    && parent.expression === self
                    && !scope_encloses_variables_in_this_scope(nearest_scope, fixed)
                    && !(fixed.name && fixed.name.definition().recursive_refs > 0);
        }

        if (single_use && fixed) {
            if (fixed instanceof AST_DefClass) {
                set_flag(fixed, SQUEEZED);
                fixed = make_node(AST_ClassExpression, fixed, fixed);
            }
            if (fixed instanceof AST_Defun) {
                set_flag(fixed, SQUEEZED);
                fixed = make_node(AST_Function, fixed, fixed);
            }
            if (def.recursive_refs > 0 && fixed.name instanceof AST_SymbolDefun) {
                const defun_def = fixed.name.definition();
                let lambda_def = fixed.variables.get(fixed.name.name);
                let name = lambda_def && lambda_def.orig[0];
                if (!(name instanceof AST_SymbolLambda)) {
                    name = make_node(AST_SymbolLambda, fixed.name, fixed.name);
                    name.scope = fixed;
                    fixed.name = name;
                    lambda_def = fixed.def_function(name);
                }
                walk(fixed, node => {
                    if (node instanceof AST_SymbolRef && node.definition() === defun_def) {
                        node.thedef = lambda_def;
                        lambda_def.references.push(node);
                    }
                });
            }
            if (
                (fixed instanceof AST_Lambda || fixed instanceof AST_Class)
                && fixed.parent_scope !== nearest_scope
            ) {
                fixed = fixed.clone(true, compressor.get_toplevel());

                nearest_scope.add_child_scope(fixed);
            }
            return fixed.optimize(compressor);
        }

        // multiple uses
        if (fixed) {
            let replace;

            if (fixed instanceof AST_This) {
                if (!(def.orig[0] instanceof AST_SymbolFunarg)
                    && def.references.every((ref) =>
                        def.scope === ref.scope
                    )) {
                    replace = fixed;
                }
            } else {
                var ev = fixed.evaluate(compressor);
                if (
                    ev !== fixed
                    && (compressor.option("unsafe_regexp") || !(ev instanceof RegExp))
                ) {
                    replace = make_node_from_constant(ev, fixed);
                }
            }

            if (replace) {
                const name_length = self.size(compressor);
                const replace_size = replace.size(compressor);

                let overhead = 0;
                if (compressor.option("unused") && !compressor.exposed(def)) {
                    overhead =
                        (name_length + 2 + replace_size) /
                        (def.references.length - def.assignments);
                }

                if (replace_size <= name_length + overhead) {
                    return replace;
                }
            }
        }
    }

    return self;
});

function scope_encloses_variables_in_this_scope(scope, pulled_scope) {
    for (const enclosed of pulled_scope.enclosed) {
        if (pulled_scope.variables.has(enclosed.name)) {
            continue;
        }
        const looked_up = scope.find_variable(enclosed.name);
        if (looked_up) {
            if (looked_up === enclosed) continue;
            return true;
        }
    }
    return false;
}

function is_atomic(lhs, self) {
    return lhs instanceof AST_SymbolRef || lhs.TYPE === self.TYPE;
}

def_optimize(AST_Undefined, function(self, compressor) {
    if (compressor.option("unsafe_undefined")) {
        var undef = find_variable(compressor, "undefined");
        if (undef) {
            var ref = make_node(AST_SymbolRef, self, {
                name   : "undefined",
                scope  : undef.scope,
                thedef : undef
            });
            set_flag(ref, UNDEFINED);
            return ref;
        }
    }
    var lhs = is_lhs(compressor.self(), compressor.parent());
    if (lhs && is_atomic(lhs, self)) return self;
    return make_node(AST_UnaryPrefix, self, {
        operator: "void",
        expression: make_node(AST_Number, self, {
            value: 0
        })
    });
});

def_optimize(AST_Infinity, function(self, compressor) {
    var lhs = is_lhs(compressor.self(), compressor.parent());
    if (lhs && is_atomic(lhs, self)) return self;
    if (
        compressor.option("keep_infinity")
        && !(lhs && !is_atomic(lhs, self))
        && !find_variable(compressor, "Infinity")
    ) {
        return self;
    }
    return make_node(AST_Binary, self, {
        operator: "/",
        left: make_node(AST_Number, self, {
            value: 1
        }),
        right: make_node(AST_Number, self, {
            value: 0
        })
    });
});

def_optimize(AST_NaN, function(self, compressor) {
    var lhs = is_lhs(compressor.self(), compressor.parent());
    if (lhs && !is_atomic(lhs, self)
        || find_variable(compressor, "NaN")) {
        return make_node(AST_Binary, self, {
            operator: "/",
            left: make_node(AST_Number, self, {
                value: 0
            }),
            right: make_node(AST_Number, self, {
                value: 0
            })
        });
    }
    return self;
});

function is_reachable(self, defs) {
    const find_ref = node => {
        if (node instanceof AST_SymbolRef && member(node.definition(), defs)) {
            return walk_abort;
        }
    };

    return walk_parent(self, (node, info) => {
        if (node instanceof AST_Scope && node !== self) {
            var parent = info.parent();

            if (parent instanceof AST_Call && parent.expression === node) return;

            if (walk(node, find_ref)) return walk_abort;

            return true;
        }
    });
}

const ASSIGN_OPS = makePredicate("+ - / * % >> << >>> | ^ &");
const ASSIGN_OPS_COMMUTATIVE = makePredicate("* | ^ &");
def_optimize(AST_Assign, function(self, compressor) {
    if (self.logical) {
        return self.lift_sequences(compressor);
    }

    var def;
    if (compressor.option("dead_code")
        && self.left instanceof AST_SymbolRef
        && (def = self.left.definition()).scope === compressor.find_parent(AST_Lambda)) {
        var level = 0, node, parent = self;
        do {
            node = parent;
            parent = compressor.parent(level++);
            if (parent instanceof AST_Exit) {
                if (in_try(level, parent)) break;
                if (is_reachable(def.scope, [ def ])) break;
                if (self.operator == "=") return self.right;
                def.fixed = false;
                return make_node(AST_Binary, self, {
                    operator: self.operator.slice(0, -1),
                    left: self.left,
                    right: self.right
                }).optimize(compressor);
            }
        } while (parent instanceof AST_Binary && parent.right === node
            || parent instanceof AST_Sequence && parent.tail_node() === node);
    }
    self = self.lift_sequences(compressor);
    if (self.operator == "=" && self.left instanceof AST_SymbolRef && self.right instanceof AST_Binary) {
        // x = expr1 OP expr2
        if (self.right.left instanceof AST_SymbolRef
            && self.right.left.name == self.left.name
            && ASSIGN_OPS.has(self.right.operator)) {
            // x = x - 2  --->  x -= 2
            self.operator = self.right.operator + "=";
            self.right = self.right.right;
        } else if (self.right.right instanceof AST_SymbolRef
            && self.right.right.name == self.left.name
            && ASSIGN_OPS_COMMUTATIVE.has(self.right.operator)
            && !self.right.left.has_side_effects(compressor)) {
            // x = 2 & x  --->  x &= 2
            self.operator = self.right.operator + "=";
            self.right = self.right.left;
        }
    }
    return self;

    function in_try(level, node) {
        var right = self.right;
        self.right = make_node(AST_Null, right);
        var may_throw = node.may_throw(compressor);
        self.right = right;
        var scope = self.left.definition().scope;
        var parent;
        while ((parent = compressor.parent(level++)) !== scope) {
            if (parent instanceof AST_Try) {
                if (parent.bfinally) return true;
                if (may_throw && parent.bcatch) return true;
            }
        }
    }
});

def_optimize(AST_DefaultAssign, function(self, compressor) {
    if (!compressor.option("evaluate")) {
        return self;
    }
    var evaluateRight = self.right.evaluate(compressor);

    // `[x = undefined] = foo` ---> `[x] = foo`
    if (evaluateRight === undefined) {
        self = self.left;
    } else if (evaluateRight !== self.right) {
        evaluateRight = make_node_from_constant(evaluateRight, self.right);
        self.right = best_of_expression(evaluateRight, self.right);
    }

    return self;
});

function is_nullish(node) {
    let fixed;
    return (
        node instanceof AST_Null
        || is_undefined(node)
        || (
            node instanceof AST_SymbolRef
            && (fixed = node.definition().fixed) instanceof AST_Node
            && is_nullish(fixed)
        )
        // Recurse into those optional chains!
        || node instanceof AST_PropAccess && node.optional && is_nullish(node.expression)
        || node instanceof AST_Call && node.optional && is_nullish(node.expression)
        || node instanceof AST_Chain && is_nullish(node.expression)
    );
}

function is_nullish_check(check, check_subject, compressor) {
    if (check_subject.may_throw(compressor)) return false;

    let nullish_side;

    // foo == null
    if (
        check instanceof AST_Binary
        && check.operator === "=="
        // which side is nullish?
        && (
            (nullish_side = is_nullish(check.left) && check.left)
            || (nullish_side = is_nullish(check.right) && check.right)
        )
        // is the other side the same as the check_subject
        && (
            nullish_side === check.left
                ? check.right
                : check.left
        ).equivalent_to(check_subject)
    ) {
        return true;
    }

    // foo === null || foo === undefined
    if (check instanceof AST_Binary && check.operator === "||") {
        let null_cmp;
        let undefined_cmp;

        const find_comparison = cmp => {
            if (!(
                cmp instanceof AST_Binary
                && (cmp.operator === "===" || cmp.operator === "==")
            )) {
                return false;
            }

            let found = 0;
            let defined_side;

            if (cmp.left instanceof AST_Null) {
                found++;
                null_cmp = cmp;
                defined_side = cmp.right;
            }
            if (cmp.right instanceof AST_Null) {
                found++;
                null_cmp = cmp;
                defined_side = cmp.left;
            }
            if (is_undefined(cmp.left)) {
                found++;
                undefined_cmp = cmp;
                defined_side = cmp.right;
            }
            if (is_undefined(cmp.right)) {
                found++;
                undefined_cmp = cmp;
                defined_side = cmp.left;
            }

            if (found !== 1) {
                return false;
            }

            if (!defined_side.equivalent_to(check_subject)) {
                return false;
            }

            return true;
        };

        if (!find_comparison(check.left)) return false;
        if (!find_comparison(check.right)) return false;

        if (null_cmp && undefined_cmp && null_cmp !== undefined_cmp) {
            return true;
        }
    }

    return false;
}

def_optimize(AST_Conditional, function(self, compressor) {
    if (!compressor.option("conditionals")) return self;
    // This looks like lift_sequences(), should probably be under "sequences"
    if (self.condition instanceof AST_Sequence) {
        var expressions = self.condition.expressions.slice();
        self.condition = expressions.pop();
        expressions.push(self);
        return make_sequence(self, expressions);
    }
    var cond = self.condition.evaluate(compressor);
    if (cond !== self.condition) {
        if (cond) {
            return maintain_this_binding(compressor.parent(), compressor.self(), self.consequent);
        } else {
            return maintain_this_binding(compressor.parent(), compressor.self(), self.alternative);
        }
    }
    var negated = cond.negate(compressor, first_in_statement(compressor));
    if (best_of(compressor, cond, negated) === negated) {
        self = make_node(AST_Conditional, self, {
            condition: negated,
            consequent: self.alternative,
            alternative: self.consequent
        });
    }
    var condition = self.condition;
    var consequent = self.consequent;
    var alternative = self.alternative;
    // x?x:y --> x||y
    if (condition instanceof AST_SymbolRef
        && consequent instanceof AST_SymbolRef
        && condition.definition() === consequent.definition()) {
        return make_node(AST_Binary, self, {
            operator: "||",
            left: condition,
            right: alternative
        });
    }
    // if (foo) exp = something; else exp = something_else;
    //                   |
    //                   v
    // exp = foo ? something : something_else;
    if (
        consequent instanceof AST_Assign
        && alternative instanceof AST_Assign
        && consequent.operator === alternative.operator
        && consequent.logical === alternative.logical
        && consequent.left.equivalent_to(alternative.left)
        && (!self.condition.has_side_effects(compressor)
            || consequent.operator == "="
                && !consequent.left.has_side_effects(compressor))
    ) {
        return make_node(AST_Assign, self, {
            operator: consequent.operator,
            left: consequent.left,
            logical: consequent.logical,
            right: make_node(AST_Conditional, self, {
                condition: self.condition,
                consequent: consequent.right,
                alternative: alternative.right
            })
        });
    }
    // x ? y(a) : y(b) --> y(x ? a : b)
    var arg_index;
    if (consequent instanceof AST_Call
        && alternative.TYPE === consequent.TYPE
        && consequent.args.length > 0
        && consequent.args.length == alternative.args.length
        && consequent.expression.equivalent_to(alternative.expression)
        && !self.condition.has_side_effects(compressor)
        && !consequent.expression.has_side_effects(compressor)
        && typeof (arg_index = single_arg_diff()) == "number") {
        var node = consequent.clone();
        node.args[arg_index] = make_node(AST_Conditional, self, {
            condition: self.condition,
            consequent: consequent.args[arg_index],
            alternative: alternative.args[arg_index]
        });
        return node;
    }
    // a ? b : c ? b : d --> (a || c) ? b : d
    if (alternative instanceof AST_Conditional
        && consequent.equivalent_to(alternative.consequent)) {
        return make_node(AST_Conditional, self, {
            condition: make_node(AST_Binary, self, {
                operator: "||",
                left: condition,
                right: alternative.condition
            }),
            consequent: consequent,
            alternative: alternative.alternative
        }).optimize(compressor);
    }

    // a == null ? b : a -> a ?? b
    if (
        compressor.option("ecma") >= 2020 &&
        is_nullish_check(condition, alternative, compressor)
    ) {
        return make_node(AST_Binary, self, {
            operator: "??",
            left: alternative,
            right: consequent
        }).optimize(compressor);
    }

    // a ? b : (c, b) --> (a || c), b
    if (alternative instanceof AST_Sequence
        && consequent.equivalent_to(alternative.expressions[alternative.expressions.length - 1])) {
        return make_sequence(self, [
            make_node(AST_Binary, self, {
                operator: "||",
                left: condition,
                right: make_sequence(self, alternative.expressions.slice(0, -1))
            }),
            consequent
        ]).optimize(compressor);
    }
    // a ? b : (c && b) --> (a || c) && b
    if (alternative instanceof AST_Binary
        && alternative.operator == "&&"
        && consequent.equivalent_to(alternative.right)) {
        return make_node(AST_Binary, self, {
            operator: "&&",
            left: make_node(AST_Binary, self, {
                operator: "||",
                left: condition,
                right: alternative.left
            }),
            right: consequent
        }).optimize(compressor);
    }
    // x?y?z:a:a --> x&&y?z:a
    if (consequent instanceof AST_Conditional
        && consequent.alternative.equivalent_to(alternative)) {
        return make_node(AST_Conditional, self, {
            condition: make_node(AST_Binary, self, {
                left: self.condition,
                operator: "&&",
                right: consequent.condition
            }),
            consequent: consequent.consequent,
            alternative: alternative
        });
    }
    // x ? y : y --> x, y
    if (consequent.equivalent_to(alternative)) {
        return make_sequence(self, [
            self.condition,
            consequent
        ]).optimize(compressor);
    }
    // x ? y || z : z --> x && y || z
    if (consequent instanceof AST_Binary
        && consequent.operator == "||"
        && consequent.right.equivalent_to(alternative)) {
        return make_node(AST_Binary, self, {
            operator: "||",
            left: make_node(AST_Binary, self, {
                operator: "&&",
                left: self.condition,
                right: consequent.left
            }),
            right: alternative
        }).optimize(compressor);
    }
    var in_bool = compressor.in_boolean_context();
    if (is_true(self.consequent)) {
        if (is_false(self.alternative)) {
            // c ? true : false ---> !!c
            return booleanize(self.condition);
        }
        // c ? true : x ---> !!c || x
        return make_node(AST_Binary, self, {
            operator: "||",
            left: booleanize(self.condition),
            right: self.alternative
        });
    }
    if (is_false(self.consequent)) {
        if (is_true(self.alternative)) {
            // c ? false : true ---> !c
            return booleanize(self.condition.negate(compressor));
        }
        // c ? false : x ---> !c && x
        return make_node(AST_Binary, self, {
            operator: "&&",
            left: booleanize(self.condition.negate(compressor)),
            right: self.alternative
        });
    }
    if (is_true(self.alternative)) {
        // c ? x : true ---> !c || x
        return make_node(AST_Binary, self, {
            operator: "||",
            left: booleanize(self.condition.negate(compressor)),
            right: self.consequent
        });
    }
    if (is_false(self.alternative)) {
        // c ? x : false ---> !!c && x
        return make_node(AST_Binary, self, {
            operator: "&&",
            left: booleanize(self.condition),
            right: self.consequent
        });
    }

    return self;

    function booleanize(node) {
        if (node.is_boolean()) return node;
        // !!expression
        return make_node(AST_UnaryPrefix, node, {
            operator: "!",
            expression: node.negate(compressor)
        });
    }

    // AST_True or !0
    function is_true(node) {
        return node instanceof AST_True
            || in_bool
                && node instanceof AST_Constant
                && node.getValue()
            || (node instanceof AST_UnaryPrefix
                && node.operator == "!"
                && node.expression instanceof AST_Constant
                && !node.expression.getValue());
    }
    // AST_False or !1
    function is_false(node) {
        return node instanceof AST_False
            || in_bool
                && node instanceof AST_Constant
                && !node.getValue()
            || (node instanceof AST_UnaryPrefix
                && node.operator == "!"
                && node.expression instanceof AST_Constant
                && node.expression.getValue());
    }

    function single_arg_diff() {
        var a = consequent.args;
        var b = alternative.args;
        for (var i = 0, len = a.length; i < len; i++) {
            if (a[i] instanceof AST_Expansion) return;
            if (!a[i].equivalent_to(b[i])) {
                if (b[i] instanceof AST_Expansion) return;
                for (var j = i + 1; j < len; j++) {
                    if (a[j] instanceof AST_Expansion) return;
                    if (!a[j].equivalent_to(b[j])) return;
                }
                return i;
            }
        }
    }
});

def_optimize(AST_Boolean, function(self, compressor) {
    if (compressor.in_boolean_context()) return make_node(AST_Number, self, {
        value: +self.value
    });
    var p = compressor.parent();
    if (compressor.option("booleans_as_integers")) {
        if (p instanceof AST_Binary && (p.operator == "===" || p.operator == "!==")) {
            p.operator = p.operator.replace(/=$/, "");
        }
        return make_node(AST_Number, self, {
            value: +self.value
        });
    }
    if (compressor.option("booleans")) {
        if (p instanceof AST_Binary && (p.operator == "=="
                                        || p.operator == "!=")) {
            return make_node(AST_Number, self, {
                value: +self.value
            });
        }
        return make_node(AST_UnaryPrefix, self, {
            operator: "!",
            expression: make_node(AST_Number, self, {
                value: 1 - self.value
            })
        });
    }
    return self;
});

function safe_to_flatten(value, compressor) {
    if (value instanceof AST_SymbolRef) {
        value = value.fixed_value();
    }
    if (!value) return false;
    if (!(value instanceof AST_Lambda || value instanceof AST_Class)) return true;
    if (!(value instanceof AST_Lambda && value.contains_this())) return true;
    return compressor.parent() instanceof AST_New;
}

AST_PropAccess.DEFMETHOD("flatten_object", function(key, compressor) {
    if (!compressor.option("properties")) return;
    if (key === "__proto__") return;

    var arrows = compressor.option("unsafe_arrows") && compressor.option("ecma") >= 2015;
    var expr = this.expression;
    if (expr instanceof AST_Object) {
        var props = expr.properties;

        for (var i = props.length; --i >= 0;) {
            var prop = props[i];

            if ("" + (prop instanceof AST_ConciseMethod ? prop.key.name : prop.key) == key) {
                const all_props_flattenable = props.every((p) =>
                    (p instanceof AST_ObjectKeyVal
                        || arrows && p instanceof AST_ConciseMethod && !p.is_generator
                    )
                    && !p.computed_key()
                );

                if (!all_props_flattenable) return;
                if (!safe_to_flatten(prop.value, compressor)) return;

                return make_node(AST_Sub, this, {
                    expression: make_node(AST_Array, expr, {
                        elements: props.map(function(prop) {
                            var v = prop.value;
                            if (v instanceof AST_Accessor) {
                                v = make_node(AST_Function, v, v);
                            }

                            var k = prop.key;
                            if (k instanceof AST_Node && !(k instanceof AST_SymbolMethod)) {
                                return make_sequence(prop, [ k, v ]);
                            }

                            return v;
                        })
                    }),
                    property: make_node(AST_Number, this, {
                        value: i
                    })
                });
            }
        }
    }
});

def_optimize(AST_Sub, function(self, compressor) {
    var expr = self.expression;
    var prop = self.property;
    if (compressor.option("properties")) {
        var key = prop.evaluate(compressor);
        if (key !== prop) {
            if (typeof key == "string") {
                if (key == "undefined") {
                    key = undefined;
                } else {
                    var value = parseFloat(key);
                    if (value.toString() == key) {
                        key = value;
                    }
                }
            }
            prop = self.property = best_of_expression(prop, make_node_from_constant(key, prop).transform(compressor));
            var property = "" + key;
            if (is_basic_identifier_string(property)
                && property.length <= prop.size() + 1) {
                return make_node(AST_Dot, self, {
                    expression: expr,
                    optional: self.optional,
                    property: property,
                    quote: prop.quote,
                }).optimize(compressor);
            }
        }
    }
    var fn;
    OPT_ARGUMENTS: if (compressor.option("arguments")
        && expr instanceof AST_SymbolRef
        && expr.name == "arguments"
        && expr.definition().orig.length == 1
        && (fn = expr.scope) instanceof AST_Lambda
        && fn.uses_arguments
        && !(fn instanceof AST_Arrow)
        && prop instanceof AST_Number) {
        var index = prop.getValue();
        var params = new Set();
        var argnames = fn.argnames;
        for (var n = 0; n < argnames.length; n++) {
            if (!(argnames[n] instanceof AST_SymbolFunarg)) {
                break OPT_ARGUMENTS; // destructuring parameter - bail
            }
            var param = argnames[n].name;
            if (params.has(param)) {
                break OPT_ARGUMENTS; // duplicate parameter - bail
            }
            params.add(param);
        }
        var argname = fn.argnames[index];
        if (argname && compressor.has_directive("use strict")) {
            var def = argname.definition();
            if (!compressor.option("reduce_vars") || def.assignments || def.orig.length > 1) {
                argname = null;
            }
        } else if (!argname && !compressor.option("keep_fargs") && index < fn.argnames.length + 5) {
            while (index >= fn.argnames.length) {
                argname = fn.create_symbol(AST_SymbolFunarg, {
                    source: fn,
                    scope: fn,
                    tentative_name: "argument_" + fn.argnames.length,
                });
                fn.argnames.push(argname);
            }
        }
        if (argname) {
            var sym = make_node(AST_SymbolRef, self, argname);
            sym.reference({});
            clear_flag(argname, UNUSED);
            return sym;
        }
    }
    if (is_lhs(self, compressor.parent())) return self;
    if (key !== prop) {
        var sub = self.flatten_object(property, compressor);
        if (sub) {
            expr = self.expression = sub.expression;
            prop = self.property = sub.property;
        }
    }
    if (compressor.option("properties") && compressor.option("side_effects")
        && prop instanceof AST_Number && expr instanceof AST_Array) {
        var index = prop.getValue();
        var elements = expr.elements;
        var retValue = elements[index];
        FLATTEN: if (safe_to_flatten(retValue, compressor)) {
            var flatten = true;
            var values = [];
            for (var i = elements.length; --i > index;) {
                var value = elements[i].drop_side_effect_free(compressor);
                if (value) {
                    values.unshift(value);
                    if (flatten && value.has_side_effects(compressor)) flatten = false;
                }
            }
            if (retValue instanceof AST_Expansion) break FLATTEN;
            retValue = retValue instanceof AST_Hole ? make_node(AST_Undefined, retValue) : retValue;
            if (!flatten) values.unshift(retValue);
            while (--i >= 0) {
                var value = elements[i];
                if (value instanceof AST_Expansion) break FLATTEN;
                value = value.drop_side_effect_free(compressor);
                if (value) values.unshift(value);
                else index--;
            }
            if (flatten) {
                values.push(retValue);
                return make_sequence(self, values).optimize(compressor);
            } else return make_node(AST_Sub, self, {
                expression: make_node(AST_Array, expr, {
                    elements: values
                }),
                property: make_node(AST_Number, prop, {
                    value: index
                })
            });
        }
    }
    var ev = self.evaluate(compressor);
    if (ev !== self) {
        ev = make_node_from_constant(ev, self).optimize(compressor);
        return best_of(compressor, ev, self);
    }
    if (self.optional && is_nullish(self.expression)) {
        return make_node(AST_Undefined, self);
    }
    return self;
});

def_optimize(AST_Chain, function (self, compressor) {
    self.expression = self.expression.optimize(compressor);
    return self;
});

AST_Lambda.DEFMETHOD("contains_this", function() {
    return walk(this, node => {
        if (node instanceof AST_This) return walk_abort;
        if (
            node !== this
            && node instanceof AST_Scope
            && !(node instanceof AST_Arrow)
        ) {
            return true;
        }
    });
});

def_optimize(AST_Dot, function(self, compressor) {
    const parent = compressor.parent();
    if (is_lhs(self, parent)) return self;
    if (compressor.option("unsafe_proto")
        && self.expression instanceof AST_Dot
        && self.expression.property == "prototype") {
        var exp = self.expression.expression;
        if (is_undeclared_ref(exp)) switch (exp.name) {
          case "Array":
            self.expression = make_node(AST_Array, self.expression, {
                elements: []
            });
            break;
          case "Function":
            self.expression = make_node(AST_Function, self.expression, {
                argnames: [],
                body: []
            });
            break;
          case "Number":
            self.expression = make_node(AST_Number, self.expression, {
                value: 0
            });
            break;
          case "Object":
            self.expression = make_node(AST_Object, self.expression, {
                properties: []
            });
            break;
          case "RegExp":
            self.expression = make_node(AST_RegExp, self.expression, {
                value: { source: "t", flags: "" }
            });
            break;
          case "String":
            self.expression = make_node(AST_String, self.expression, {
                value: ""
            });
            break;
        }
    }
    if (!(parent instanceof AST_Call) || !has_annotation(parent, _NOINLINE)) {
        const sub = self.flatten_object(self.property, compressor);
        if (sub) return sub.optimize(compressor);
    }
    let ev = self.evaluate(compressor);
    if (ev !== self) {
        ev = make_node_from_constant(ev, self).optimize(compressor);
        return best_of(compressor, ev, self);
    }
    if (self.optional && is_nullish(self.expression)) {
        return make_node(AST_Undefined, self);
    }
    return self;
});

function literals_in_boolean_context(self, compressor) {
    if (compressor.in_boolean_context()) {
        return best_of(compressor, self, make_sequence(self, [
            self,
            make_node(AST_True, self)
        ]).optimize(compressor));
    }
    return self;
}

function inline_array_like_spread(elements) {
    for (var i = 0; i < elements.length; i++) {
        var el = elements[i];
        if (el instanceof AST_Expansion) {
            var expr = el.expression;
            if (
                expr instanceof AST_Array
                && !expr.elements.some(elm => elm instanceof AST_Hole)
            ) {
                elements.splice(i, 1, ...expr.elements);
                // Step back one, as the element at i is now new.
                i--;
            }
            // In array-like spread, spreading a non-iterable value is TypeError.
            // We therefore can’t optimize anything else, unlike with object spread.
        }
    }
}

def_optimize(AST_Array, function(self, compressor) {
    var optimized = literals_in_boolean_context(self, compressor);
    if (optimized !== self) {
        return optimized;
    }
    inline_array_like_spread(self.elements);
    return self;
});

function inline_object_prop_spread(props) {
    for (var i = 0; i < props.length; i++) {
        var prop = props[i];
        if (prop instanceof AST_Expansion) {
            const expr = prop.expression;
            if (
                expr instanceof AST_Object
                && expr.properties.every(prop => prop instanceof AST_ObjectKeyVal)
            ) {
                props.splice(i, 1, ...expr.properties);
                // Step back one, as the property at i is now new.
                i--;
            } else if (expr instanceof AST_Constant
                && !(expr instanceof AST_String)) {
                // Unlike array-like spread, in object spread, spreading a
                // non-iterable value silently does nothing; it is thus safe
                // to remove. AST_String is the only iterable AST_Constant.
                props.splice(i, 1);
            }
        }
    }
}

def_optimize(AST_Object, function(self, compressor) {
    var optimized = literals_in_boolean_context(self, compressor);
    if (optimized !== self) {
        return optimized;
    }
    inline_object_prop_spread(self.properties);
    return self;
});

def_optimize(AST_RegExp, literals_in_boolean_context);

def_optimize(AST_Return, function(self, compressor) {
    if (self.value && is_undefined(self.value, compressor)) {
        self.value = null;
    }
    return self;
});

def_optimize(AST_Arrow, opt_AST_Lambda);

def_optimize(AST_Function, function(self, compressor) {
    self = opt_AST_Lambda(self, compressor);
    if (compressor.option("unsafe_arrows")
        && compressor.option("ecma") >= 2015
        && !self.name
        && !self.is_generator
        && !self.uses_arguments
        && !self.pinned()) {
        const uses_this = walk(self, node => {
            if (node instanceof AST_This) return walk_abort;
        });
        if (!uses_this) return make_node(AST_Arrow, self, self).optimize(compressor);
    }
    return self;
});

def_optimize(AST_Class, function(self) {
    // HACK to avoid compress failure.
    // AST_Class is not really an AST_Scope/AST_Block as it lacks a body.
    return self;
});

def_optimize(AST_Yield, function(self, compressor) {
    if (self.expression && !self.is_star && is_undefined(self.expression, compressor)) {
        self.expression = null;
    }
    return self;
});

def_optimize(AST_TemplateString, function(self, compressor) {
    if (
        !compressor.option("evaluate")
        || compressor.parent() instanceof AST_PrefixedTemplateString
    ) {
        return self;
    }

    var segments = [];
    for (var i = 0; i < self.segments.length; i++) {
        var segment = self.segments[i];
        if (segment instanceof AST_Node) {
            var result = segment.evaluate(compressor);
            // Evaluate to constant value
            // Constant value shorter than ${segment}
            if (result !== segment && (result + "").length <= segment.size() + "${}".length) {
                // There should always be a previous and next segment if segment is a node
                segments[segments.length - 1].value = segments[segments.length - 1].value + result + self.segments[++i].value;
                continue;
            }
            // `before ${`innerBefore ${any} innerAfter`} after` => `before innerBefore ${any} innerAfter after`
            // TODO:
            // `before ${'test' + foo} after` => `before innerBefore ${any} innerAfter after`
            // `before ${foo + 'test} after` => `before innerBefore ${any} innerAfter after`
            if (segment instanceof AST_TemplateString) {
                var inners = segment.segments;
                segments[segments.length - 1].value += inners[0].value;
                for (var j = 1; j < inners.length; j++) {
                    segment = inners[j];
                    segments.push(segment);
                }
                continue;
            }
        }
        segments.push(segment);
    }
    self.segments = segments;

    // `foo` => "foo"
    if (segments.length == 1) {
        return make_node(AST_String, self, segments[0]);
    }

    if (
        segments.length === 3
        && segments[1] instanceof AST_Node
        && (
            segments[1].is_string(compressor)
            || segments[1].is_number(compressor)
            || is_nullish(segments[1])
            || compressor.option("unsafe")
        )
    ) {
        // `foo${bar}` => "foo" + bar
        if (segments[2].value === "") {
            return make_node(AST_Binary, self, {
                operator: "+",
                left: make_node(AST_String, self, {
                    value: segments[0].value,
                }),
                right: segments[1],
            });
        }
        // `${bar}baz` => bar + "baz"
        if (segments[0].value === "") {
            return make_node(AST_Binary, self, {
                operator: "+",
                left: segments[1],
                right: make_node(AST_String, self, {
                    value: segments[2].value,
                }),
            });
        }
    }
    return self;
});

def_optimize(AST_PrefixedTemplateString, function(self) {
    return self;
});

// ["p"]:1 ---> p:1
// [42]:1 ---> 42:1
function lift_key(self, compressor) {
    if (!compressor.option("computed_props")) return self;
    // save a comparison in the typical case
    if (!(self.key instanceof AST_Constant)) return self;
    // allow certain acceptable props as not all AST_Constants are true constants
    if (self.key instanceof AST_String || self.key instanceof AST_Number) {
        if (self.key.value === "__proto__") return self;
        if (self.key.value == "constructor"
            && compressor.parent() instanceof AST_Class) return self;
        if (self instanceof AST_ObjectKeyVal) {
            self.key = self.key.value;
        } else if (self instanceof AST_ClassProperty) {
            self.key = make_node(AST_SymbolClassProperty, self.key, {
                name: self.key.value
            });
        } else {
            self.key = make_node(AST_SymbolMethod, self.key, {
                name: self.key.value
            });
        }
    }
    return self;
}

def_optimize(AST_ObjectProperty, lift_key);

def_optimize(AST_ConciseMethod, function(self, compressor) {
    lift_key(self, compressor);
    // p(){return x;} ---> p:()=>x
    if (compressor.option("arrows")
        && compressor.parent() instanceof AST_Object
        && !self.is_generator
        && !self.value.uses_arguments
        && !self.value.pinned()
        && self.value.body.length == 1
        && self.value.body[0] instanceof AST_Return
        && self.value.body[0].value
        && !self.value.contains_this()) {
        var arrow = make_node(AST_Arrow, self.value, self.value);
        arrow.async = self.async;
        arrow.is_generator = self.is_generator;
        return make_node(AST_ObjectKeyVal, self, {
            key: self.key instanceof AST_SymbolMethod ? self.key.name : self.key,
            value: arrow,
            quote: self.quote,
        });
    }
    return self;
});

def_optimize(AST_ObjectKeyVal, function(self, compressor) {
    lift_key(self, compressor);
    // p:function(){} ---> p(){}
    // p:function*(){} ---> *p(){}
    // p:async function(){} ---> async p(){}
    // p:()=>{} ---> p(){}
    // p:async()=>{} ---> async p(){}
    var unsafe_methods = compressor.option("unsafe_methods");
    if (unsafe_methods
        && compressor.option("ecma") >= 2015
        && (!(unsafe_methods instanceof RegExp) || unsafe_methods.test(self.key + ""))) {
        var key = self.key;
        var value = self.value;
        var is_arrow_with_block = value instanceof AST_Arrow
            && Array.isArray(value.body)
            && !value.contains_this();
        if ((is_arrow_with_block || value instanceof AST_Function) && !value.name) {
            return make_node(AST_ConciseMethod, self, {
                async: value.async,
                is_generator: value.is_generator,
                key: key instanceof AST_Node ? key : make_node(AST_SymbolMethod, self, {
                    name: key,
                }),
                value: make_node(AST_Accessor, value, value),
                quote: self.quote,
            });
        }
    }
    return self;
});

def_optimize(AST_Destructuring, function(self, compressor) {
    if (compressor.option("pure_getters") == true
        && compressor.option("unused")
        && !self.is_array
        && Array.isArray(self.names)
        && !is_destructuring_export_decl(compressor)
        && !(self.names[self.names.length - 1] instanceof AST_Expansion)) {
        var keep = [];
        for (var i = 0; i < self.names.length; i++) {
            var elem = self.names[i];
            if (!(elem instanceof AST_ObjectKeyVal
                && typeof elem.key == "string"
                && elem.value instanceof AST_SymbolDeclaration
                && !should_retain(compressor, elem.value.definition()))) {
                keep.push(elem);
            }
        }
        if (keep.length != self.names.length) {
            self.names = keep;
        }
    }
    return self;

    function is_destructuring_export_decl(compressor) {
        var ancestors = [/^VarDef$/, /^(Const|Let|Var)$/, /^Export$/];
        for (var a = 0, p = 0, len = ancestors.length; a < len; p++) {
            var parent = compressor.parent(p);
            if (!parent) return false;
            if (a === 0 && parent.TYPE == "Destructuring") continue;
            if (!ancestors[a].test(parent.TYPE)) {
                return false;
            }
            a++;
        }
        return true;
    }

    function should_retain(compressor, def) {
        if (def.references.length) return true;
        if (!def.global) return false;
        if (compressor.toplevel.vars) {
             if (compressor.top_retain) {
                 return compressor.top_retain(def);
             }
             return false;
        }
        return true;
    }
});

export {
    Compressor,
};
